<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

use CBLib\Language\CBTxt;
use CBLib\Application\Application;
use CB\Database\Table\TabTable;
use CB\Database\Table\UserTable;
use CB\Plugin\Invites\Table\InviteTable;
use CB\Plugin\Invites\CBInvites;

if ( ! ( defined( '_VALID_CB' ) || defined( '_JEXEC' ) || defined( '_VALID_MOS' ) ) ) { die( 'Direct Access to this location is not allowed.' ); }

class CBplug_cbinvites extends cbPluginHandler
{

	/**
	 * @param null      $tab
	 * @param UserTable $user
	 * @param int       $ui
	 * @param array     $postdata
	 */
	public function getCBpluginComponent( $tab, $user, $ui, $postdata )
	{
		global $_CB_framework;

		outputCbJs();
		outputCbTemplate();

		$function	=	$this->getInput()->getString( 'func' );

		if ( $user === null ) {
			$user	=	CBuser::getMyUserDataInstance();
		}

		if ( $function === 'accept' ) {
			// Need to process invite acceptance now since everything that follows is for invite management:
			$this->acceptInvite( $user );
		}

		if ( ! $user->getInt( 'id', 0 ) ) {
			cbRedirect( 'index.php', CBTxt::T( 'Not authorized.' ), 'error' );
		}

		$id			=	$this->getInput()->getInt( 'id', 0 );

		if ( $tab === null ) {
			$tab	=	new TabTable();

			$tab->load( array( 'pluginclass' => 'cbinvitesTab' ) );
		}

		$profileUrl	=	$_CB_framework->userProfileUrl( $user->getInt( 'id', 0 ), false, 'cbinvitesTab' );

		if ( ! ( $tab->getBool( 'enabled', true ) && Application::MyUser()->canViewAccessLevel( $tab->getInt( 'viewaccesslevel', 1 ) ) ) ) {
			cbRedirect( $profileUrl, CBTxt::T( 'Not authorized.' ), 'error' );
		}

		ob_start();
		switch ( $function ) {
			case 'link':
				$this->createInviteLink( $user );
				break;
			case 'email':
				cbSpoofCheck( 'plugin' );
				$this->createInviteEmail( $user );
				break;
			case 'send':
				$this->sendInvite( $id, $user );
				break;
			case 'delete':
				$this->deleteInvite( $id, $user );
				break;
			default:
				cbRedirect( $profileUrl );
				break;
		}
		$html		=	ob_get_clean();

		$class		=	$this->params->getString( 'general_class' );

		echo	'<div class="cbInvites' . ( $class ? ' ' . htmlspecialchars( $class ) : null ) . '">'
			.		$html
			.	'</div>';
	}

	/**
	 * @param UserTable $user
	 */
	private function acceptInvite( $user )
	{
		global $_CB_framework, $_PLUGINS;

		if ( $user->getInt( 'id', 0 ) ) {
			cbRedirect( $_CB_framework->userProfileUrl( $user->getInt( 'id', 0 ), false, 'cbinvitesTab' ), CBTxt::T( 'You are already registered and cannot accept invites.' ), 'error' );
		}

		$code			=	$this->getInput()->getString( 'code' );

		if ( ! $code ) {
			cbRedirect( 'index.php', CBTxt::T( 'Your invite link is missing an invite code.' ), 'error' );
		}

		if ( strlen( $code ) !== 32 ) {
			cbRedirect( 'index.php', CBTxt::T( 'Your invite link does not appear to be valid.' ), 'error' );
		}

		$row			=	new InviteTable();

		$row->load( array( 'code' => $code ) );

		if ( ! $row->getInt( 'id', 0 ) ) {
			cbRedirect( 'index.php', CBTxt::T( 'Your invite link does not appear to be valid.' ), 'error' );
		}

		if ( $row->isAccepted() ) {
			cbRedirect( 'index.php', CBTxt::T( 'Your invite link has already been used.' ), 'error' );
		}

		if ( $row->isExpired() ) {
			cbRedirect( 'index.php', CBTxt::T( 'Your invite link has expired.' ), 'error' );
		}

		Application::Session()->set( 'invite_code', $code );

		$_PLUGINS->trigger( 'invites_onInviteAccept', array( $row, $code, $user ) );

		cbRedirect( $_CB_framework->viewUrl( 'registers' ), CBTxt::T( 'INVITE_LINK_ACCEPTED', '' ) );
	}

	/**
	 * @param UserTable $user
	 */
	private function createInviteLink( $user )
	{
		global $_CB_framework, $_PLUGINS;

		$profileUrl		=	$_CB_framework->userProfileUrl( $user->getInt( 'id', 0 ), false, 'cbinvitesTab' );

		if ( ! CBInvites::canCreateInvite( $user, 'links' ) ) {
			cbRedirect( $profileUrl, CBTxt::T( 'You do not have sufficient permissions to create invite links.' ), 'error' );
		}

		$row			=	new InviteTable();

		if ( Application::MyUser()->isGlobalModerator() ) {
			$row->set( 'user_id', $this->getInput()->getInt( 'get/user', $user->getInt( 'id', 0 ) ) );
		} else {
			$row->set( 'user_id', $user->getInt( 'id', 0 ) );
		}

		$row->set( 'sent', Application::Database()->getUtcDateTime() );

		$_PLUGINS->trigger( 'invites_onBeforeInvite', array( &$row, $user ) );

		if ( $row->getError() || ( ! $row->check() ) ) {
			cbRedirect( $profileUrl, CBTxt::T( 'INVITE_FAILED_SAVE_ERROR', 'Invite failed to save! [error]', array( '[error]' => $row->getError() ) ), 'error' );
		}

		if ( $row->getError() || ( ! $row->store() ) ) {
			cbRedirect( $profileUrl, CBTxt::T( 'INVITE_FAILED_SAVE_ERROR', 'Invite failed to save! [error]', array( '[error]' => $row->getError() ) ), 'error' );
		}

		$_PLUGINS->trigger( 'invites_onAfterInvite', array( $row, 0, $user ) );

		cbRedirect( $profileUrl, CBTxt::T( 'Invite link created successfully!' ) );
	}

	/**
	 * @param UserTable $user
	 */
	private function createInviteEmail( $user )
	{
		global $_CB_framework, $_PLUGINS;

		$profileUrl			=	$_CB_framework->userProfileUrl( $user->getInt( 'id', 0 ), false, 'cbinvitesTab' );

		if ( ! CBInvites::canCreateInvite( $user ) ) {
			cbRedirect( $profileUrl, CBTxt::T( 'You do not have sufficient permissions to send invites.' ), 'error' );
		}

		$cbModerator		=	Application::MyUser()->isGlobalModerator();

		if ( $this->params->getBool( 'invite_captcha', false ) && ( ! $cbModerator ) ) {
			$_PLUGINS->loadPluginGroup( 'user' );

			$_PLUGINS->trigger( 'onCheckCaptchaHtmlElements', array() );

			if ( $_PLUGINS->is_errors() ) {
				cbRedirect( $profileUrl, CBTxt::T( $_PLUGINS->getErrorMSG() ), 'error' );
			}
		}

		$toArray			=	explode( ',', $this->getInput()->getString( 'post/to' ) );

		if ( ( ! $this->params->getBool( 'invite_multiple', true ) ) && ( ! $cbModerator ) && ( count( $toArray ) > 1 ) ) {
			cbRedirect( $profileUrl, CBTxt::T( 'Multiple email addresses are not supported! Please provide a single email address to invite.' ), 'error' );
		}

		if ( empty( $toArray ) ) {
			cbRedirect( $profileUrl, CBTxt::T( 'No email address specified! Please provide a valid email address to invite.' ), 'error' );
		}

		$sent				=	0;

		foreach ( $toArray as $to ) {
			$row			=	new InviteTable();

			if ( $cbModerator ) {
				$row->set( 'user_id', $this->getInput()->getInt( 'post/user_id', $user->getInt( 'id', 0 ) ) );
			} else {
				$row->set( 'user_id', $user->getInt( 'id', 0 ) );
			}

			$row->set( 'to', trim( $to ) );

			if ( ! $row->getString( 'to' ) ) {
				continue;
			}

			if ( CBInvites::getCreateLimited( $user ) ) {
				$_CB_framework->enqueueMessage( CBTxt::T( 'INVITE_LIMIT_REACHED', '[to] could not be sent as you have reached your invite limit.', array( '[to]' => $row->getString( 'to' ) ) ), 'error' );
				continue;
			}

			$toUser			=	new UserTable();

			$toUser->loadByEmail( $row->getString( 'to' ) );

			if ( ! cbIsValidEmail( $row->getString( 'to' ) ) ) {
				$row->setError( CBTxt::T( 'INVITE_ADDRESS_INVALID', '[to] is not a valid email address.', array( '[to]' => $row->getString( 'to' ) ) ) );
			} elseif ( $user->getString( 'email' ) === $row->getString( 'to' ) || ( $toUser->getInt( 'id', 0 ) === $row->getInt( 'user_id', 0 ) ) ) {
				$row->setError( CBTxt::T( 'You cannot invite yourself.' ) );
			} elseif ( $toUser->getInt( 'id', 0 ) ) {
				$row->setError( CBTxt::T( 'INVITE_ADDRESS_ALREADY_USER', '[to] address is already a user.', array( '[to]' => $row->getString( 'to' ) ) ) );
			} elseif ( ( ! $this->params->getBool( 'invite_duplicate', false ) ) && ( ! $cbModerator ) && $row->isDuplicate() ) {
				$row->setError( CBTxt::T( 'INVITE_ADDRESS_ALREADY_INVITED', '[to] address is already invited.', array( '[to]' => $row->getString( 'to' ) ) ) );
			} elseif ( $row->isDuplicate( $user->getInt( 'id', 0 ) ) ) {
				$row->setError( CBTxt::T( 'INVITE_YOU_ALREADY_INVITED', 'You have already invited [to].', array( '[to]' => $row->getString( 'to' ) ) ) );
			}

			$_PLUGINS->trigger( 'invites_onBeforeInvite', array( &$row, $user ) );

			if ( $row->getError() || ( ! $row->check() ) ) {
				$_CB_framework->enqueueMessage( CBTxt::T( 'INVITE_FAILED_SAVE_ERROR', 'Invite failed to save! [error]', array( '[error]' => $row->getError() ) ), 'error' );
				continue;
			}

			if ( $row->getError() || ( ! $row->store() ) ) {
				$_CB_framework->enqueueMessage( CBTxt::T( 'INVITE_FAILED_SAVE_ERROR', 'Invite failed to save! [error]', array( '[error]' => $row->getError() ) ), 'error' );
				continue;
			}

			if ( ! $row->send() ) {
				$_CB_framework->enqueueMessage( CBTxt::T( 'INVITE_FAILED_SEND_ERROR', 'Invite failed to send! [error]', array( '[error]' => $row->getError() ) ), 'error' );
				continue;
			}

			$sent++;

			$_PLUGINS->trigger( 'invites_onAfterInvite', array( $row, $sent, $user ) );
		}

		cbRedirect( $profileUrl, ( $sent ? ( $sent > 1 ? CBTxt::T( 'Invites sent successfully!' ) : CBTxt::T( 'Invite sent successfully!' ) ) : null ) );
	}

	/**
	 * @param int       $id
	 * @param UserTable $user
	 */
	private function sendInvite( $id, $user )
	{
		global $_CB_framework;

		if ( ! $id ) {
			cbRedirect( $_CB_framework->userProfileUrl( $user->getInt( 'id', 0 ), false, 'cbinvitesTab' ), CBTxt::T( 'Not authorized.' ), 'error' );
		}

		$row			=	new InviteTable();

		$row->load( $id );

		$profileUrl		=	$_CB_framework->userProfileUrl( $row->getInt( 'user_id', $user->getInt( 'id', 0 ) ), false, 'cbinvitesTab' );

		if ( ( ! $row->getInt( 'id', 0 ) ) || ( ( $row->getInt( 'user_id', 0 ) !== $user->getInt( 'id', 0 ) ) && ( ! Application::MyUser()->isGlobalModerator() ) ) ) {
			cbRedirect( $profileUrl, CBTxt::T( 'Not authorized.' ), 'error' );
		}

		if ( $row->isAccepted() ) {
			cbRedirect( $profileUrl, CBTxt::T( 'Invite already accepted and cannot be resent.' ), 'error' );
		}

		if ( ! $row->canResend() ) {
			cbRedirect( $profileUrl, CBTxt::T( 'Invite resend not applicable at this time.' ), 'error' );
		}

		if ( ! $row->send() ) {
			cbRedirect( $profileUrl, CBTxt::T( 'INVITE_FAILED_SEND_ERROR', 'Invite failed to send! Error: [error]', array( '[error]' => $row->getError() ) ), 'error' );
		}

		cbRedirect( $profileUrl, CBTxt::T( 'Invite sent successfully!' ) );
	}

	/**
	 * @param int       $id
	 * @param UserTable $user
	 */
	private function deleteInvite( $id, $user )
	{
		global $_CB_framework;

		if ( ! $id ) {
			cbRedirect( $_CB_framework->userProfileUrl( $user->getInt( 'id', 0 ), false, 'cbinvitesTab' ), CBTxt::T( 'Not authorized.' ), 'error' );
		}

		$row			=	new InviteTable();

		$row->load( $id );

		$profileUrl		=	$_CB_framework->userProfileUrl( $row->getInt( 'user_id', $user->getInt( 'id', 0 ) ), false, 'cbinvitesTab' );

		if ( ( ! $row->getInt( 'id', 0 ) ) || ( ( $row->getInt( 'user_id', 0 ) !== $user->getInt( 'id', 0 ) ) && ( ! Application::MyUser()->isGlobalModerator() ) ) ) {
			cbRedirect( $profileUrl, CBTxt::T( 'Not authorized.' ), 'error' );
		}

		if ( $row->isAccepted() ) {
			cbRedirect( $profileUrl, CBTxt::T( 'Invite already accepted and cannot be deleted.' ), 'error' );
		}

		if ( $row->getError() || ( ! $row->canDelete() ) ) {
			cbRedirect( $profileUrl, CBTxt::T( 'INVITE_FAILED_DELETE_ERROR', 'Invite failed to delete! Error: [error]', array( '[error]' => $row->getError() ) ), 'error' );
		}

		if ( $row->getError() || ( ! $row->delete() ) ) {
			cbRedirect( $profileUrl, CBTxt::T( 'INVITE_FAILED_DELETE_ERROR', 'Invite failed to delete! Error: [error]', array( '[error]' => $row->getError() ) ), 'error' );
		}

		cbRedirect( $profileUrl, CBTxt::T( 'Invite deleted successfully!' ) );
	}
}
