<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\AutoActions\Action;

use CB\Database\Table\UserTable;
use CBLib\Application\Application;
use CBLib\Language\CBTxt;

\defined( 'CBLIB' ) or die();

class AcymailingAction extends Action
{

	/**
	 * @param UserTable $user
	 */
	public function execute( UserTable $user ): void
	{
		global $_CB_framework;

		if ( ! $this->installed() ) {
			$this->error( CBTxt::T( 'AUTO_ACTION_ACYMAILING_NOT_INSTALLED', ':: Action [action] :: AcyMailing is not installed', [ '[action]' => $this->autoaction()->getId() ] ) );
			return;
		}

		$params								=	$this->autoaction()->getParams()->subTree( 'acymailing' );
		$forUser							=	$params->getString( 'for', '' );

		if ( ! $forUser ) {
			$forUser						=	$user->getInt( 'id', 0 );
		} else {
			$forUser						=	$this->string( $user, $forUser );
		}

		if ( ( $user->getInt( 'id', 0 ) !== $forUser ) || ( $user->getString( 'email', '' ) !== $forUser ) ) {
			if ( is_numeric( $forUser ) ) {
				$actionUser					=	\CBuser::getUserDataInstance( $forUser );

				$forUser					=	$actionUser->getInt( 'id', 0 );
			} else {
				$actionUser					=	new UserTable();

				$actionUser->loadByEmail( $forUser );
			}
		} else {
			$actionUser						=	$user;

			$forUser						=	$user->getInt( 'id', 0 );
		}

		if ( ! $forUser ) {
			$this->error( CBTxt::T( 'AUTO_ACTION_ACYMAILING_NO_USER', ':: Action [action] :: AcyMailing skipped due to missing user', [ '[action]' => $this->autoaction()->getId() ] ) );
			return;
		}

		if ( $this->version() === 6 ) {
			require_once( $_CB_framework->getCfg( 'absolute_path' ) . '/administrator/components/com_acym/helpers/helper.php' );

			/** @var \acymuserClass $acySubscriberAPI */
			$acySubscriberAPI				=	acym_get( 'class.user' );
			$subscriber						=	$acySubscriberAPI->getOneByCMSId( $forUser );
			$subscriberId					=	( $subscriber ? (int) $subscriber->id : 0 );
		} else {
			require_once( $_CB_framework->getCfg( 'absolute_path' ) . '/administrator/components/com_acymailing/helpers/helper.php' );

			/** @var \subscriberClass $acySubscriberAPI */
			$acySubscriberAPI				=	acymailing_get( 'class.subscriber' );
			$subscriberId					=	$acySubscriberAPI->subid( $forUser );
		}

		if ( ! $subscriberId ) {
			$newSubscriber					=	new \stdClass();

			if ( $actionUser->getInt( 'id', 0 ) ) {
				$newSubscriber->name		=	$actionUser->getString( 'name', '' );
				$newSubscriber->email		=	$actionUser->getString( 'email', '' );

				if ( $this->version() === 6 ) {
					$newSubscriber->cms_id	=	$actionUser->getInt( 'id', 0 );
					$newSubscriber->source	=	'community_builder';
				} else {
					$newSubscriber->userid	=	$actionUser->getInt( 'id', 0 );
					$newSubscriber->created	=	Application::Date( $actionUser->getString( 'registerDate', '' ), 'UTC' )->getTimestamp();
					$newSubscriber->ip		=	$actionUser->getString( 'registeripaddr', '' );
				}
			} else {
				$ipAddresses				=	cbGetIParray();

				$newSubscriber->email		=	$forUser;

				if ( $this->version() === 6 ) {
					$newSubscriber->source	=	'community_builder';
				} else {
					$newSubscriber->created	=	Application::Date( 'now', 'UTC' )->getTimestamp();
					$newSubscriber->ip		=	trim( array_shift( $ipAddresses ) );
				}
			}

			if ( $this->version() === 6 ) {
				$newSubscriber->active		=	1;
			} else {
				$newSubscriber->enabled		=	1;
				$newSubscriber->accept		=	1;
				$newSubscriber->html		=	1;
			}

			$newSubscriber->confirmed		=	1;

			$subscriberId					=	$acySubscriberAPI->save( $newSubscriber );
		}

		if ( ! $subscriberId ) {
			$this->error( CBTxt::T( 'AUTO_ACTION_ACYMAILING_NO_SUB', ':: Action [action] :: AcyMailing skipped due to missing subscriber id', [ '[action]' => $this->autoaction()->getId() ] ) );
			return;
		}

		$lists								=	[];
		$subscribe							=	$params->getString( 'subscribe', '' );

		if ( $subscribe ) {
			$subscribe						=	cbToArrayOfInt( explode( '|*|', $subscribe ) );

			if ( $this->version() === 6 ) {
				$acySubscriberAPI->subscribe( $subscriberId, $subscribe );
			} else {
				foreach ( $subscribe as $listId ) {
					$lists[$listId]			=	[ 'status' => 1 ];
				}
			}
		}

		$unsubscribe						=	$params->getString( 'unsubscribe', '' );

		if ( $unsubscribe ) {
			$unsubscribe					=	cbToArrayOfInt( explode( '|*|', $unsubscribe ) );

			if ( $this->version() === 6 ) {
				$acySubscriberAPI->unsubscribe( $subscriberId, $unsubscribe );
			} else {
				foreach ( $unsubscribe as $listId ) {
					$lists[$listId]			=	[ 'status' => -1 ];
				}
			}
		}

		$remove								=	$params->getString( 'remove', '' );

		if ( $remove ) {
			$remove							=	cbToArrayOfInt( explode( '|*|', $remove ) );

			if ( $this->version() === 6 ) {
				$acySubscriberAPI->removeSubscription( $subscriberId, $remove );
			} else {
				foreach ( $remove as $listId ) {
					$lists[$listId]			=	[ 'status' => 0 ];
				}
			}
		}

		if ( $this->version() !== 6 ) {
			$pending						=	$params->getString( 'pending', '' );

			if ( $pending ) {
				$pending					=	cbToArrayOfInt( explode( '|*|', $pending ) );

				foreach ( $pending as $listId ) {
					$lists[$listId]			=	[ 'status' => 2 ];
				}
			}

			if ( $lists ) {
				$acySubscriberAPI->saveSubscription( $subscriberId, $lists );
			}
		}
	}

	/**
	 * @return array
	 */
	public function lists(): array
	{
		global $_CB_framework;

		$lists						=	[];

		if ( $this->installed() ) {
			if ( $this->version() === 6 ) {
				require_once( $_CB_framework->getCfg( 'absolute_path' ) . '/administrator/components/com_acym/helpers/helper.php' );

				/** @var \acymlistClass $acyListAPI */
				$acyListAPI			=	acym_get( 'class.list' );
				$acyLists			=	$acyListAPI->getAll();

				if ( $acyLists ) {
					foreach ( $acyLists as $acyList ) {
						$lists[]	=	\moscomprofilerHTML::makeOption( (string) $acyList->id, $acyList->name );
					}
				}
			} else {
				require_once( $_CB_framework->getCfg( 'absolute_path' ) . '/administrator/components/com_acymailing/helpers/helper.php' );

				/** @var \listClass $acyListAPI */
				$acyListAPI			=	acymailing_get( 'class.list' );
				$acyLists			=	$acyListAPI->getLists();

				if ( $acyLists ) {
					foreach ( $acyLists as $acyList ) {
						$lists[]	=	\moscomprofilerHTML::makeOption( (string) $acyList->listid, $acyList->name );
					}
				}
			}
		}

		return $lists;
	}

	/**
	 * @return bool
	 */
	public function installed(): bool
	{
		if ( $this->version() ) {
			return true;
		}

		return false;
	}

	/**
	 * @return int
	 */
	public function version(): int
	{
		global $_CB_framework;

		if ( file_exists( $_CB_framework->getCfg( 'absolute_path' ) . '/administrator/components/com_acym/helpers/helper.php' ) ) {
			return 6;
		}

		if ( file_exists( $_CB_framework->getCfg( 'absolute_path' ) . '/administrator/components/com_acymailing/helpers/helper.php' ) ) {
			return 5;
		}

		return 0;
	}
}