<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\Gallery\Trigger;

use CB\Plugin\Gallery\Table\FolderTable;
use CB\Plugin\Gallery\Table\ItemTable;
use CBLib\Application\Application;
use CB\Plugin\Gallery\CBGallery;
use CB\Plugin\Gallery\Gallery;
use CB\Plugin\Activity\Table\ActivityTable;
use CB\Plugin\Activity\Table\NotificationTable;
use CB\Plugin\Activity\Table\CommentTable;
use CB\Plugin\Activity\Activity;
use CB\Plugin\Activity\Notifications;
use CB\Plugin\Activity\NotificationsInterface;
use CB\Plugin\Activity\Comments;
use CB\Database\Table\UserTable;

defined('CBLIB') or die();

class ActivityTrigger extends \cbPluginHandler
{

	/**
	 * @return bool
	 */
	private function isCompatible()
	{
		global $_PLUGINS;

		static $compatible		=	null;

		if ( $compatible === null ) {
			$plugin				=	$_PLUGINS->getLoadedPlugin( 'user', 'cbactivity' );

			if ( $plugin ) {
				$pluginVersion	=	str_replace( '+build.', '+', $_PLUGINS->getPluginVersion( $plugin, true ) );

				$compatible		=	( version_compare( $pluginVersion, '4.0.0', '>=' ) && version_compare( $pluginVersion, '6.0.0', '<' ) );
			}
		}

		return $compatible;
	}

	/**
	 * @param int               $profileId
	 * @param null|string|array $asset
	 * @return Gallery|null
	 */
	private function activityGallery( $profileId, $asset = null )
	{
		global $_PLUGINS;

		if ( ! $profileId ) {
			return null;
		}

		if ( ! $asset ) {
			$asset								=	'uploads,profile.' . $profileId . ',profile.' . $profileId . '.%';
		}

		$userId									=	Application::MyUser()->getUserId();

		static $galleries						=	array();

		if ( ! isset( $galleries[$profileId][$asset][$userId] ) ) {
			$galleryId							=	Application::Input()->getString( 'post/gallery' );
			$gallery							=	new Gallery( $asset, $profileId );

			$gallery->set( 'location', CBGallery::getReturn( false, true ) );

			if ( $galleryId ) {
				$gallery->load( $galleryId );
			}

			$access								=	true;

			if ( preg_match( '/^profile(?:\.(\d+)(?:\.field\.(\d+))?)?/', $gallery->asset(), $matches ) || $gallery->getInt( 'tab', 0 ) || $gallery->getInt( 'field', 0 ) ) {
				$profileId						=	( isset( $matches[1] ) ? (int) $matches[1] : $gallery->user()->getInt( 'id', 0 ) );
				$fieldId						=	( isset( $matches[2] ) ? (int) $matches[2] : $gallery->getInt( 'field', 0 ) );
				$tabId							=	$gallery->getInt( 'tab', 0 );

				if ( $profileId !== $gallery->user()->getInt( 'id', 0 ) ) {
					$gallery->user( $profileId );
				}

				if ( $fieldId ) {
					$field						=	CBGallery::getField( $fieldId, $profileId );

					if ( ! $field ) {
						$access					=	false;
					} else {
						// ALWAYS reload field params to prevent gallery going out of sync:
						$gallery->set( 'field', $field->getInt( 'fieldid', 0 ) );

						$gallery->parse( $field->params, 'gallery_', false );
					}
				} else {
					$tab						=	CBGallery::getTab( $tabId, $profileId );

					if ( ! $tab ) {
						if ( ! in_array( 'all', $gallery->assets(), true ) ) {
							$access				=	false;
						}
					} else {
						// ALWAYS reload tab params to prevent gallery going out of sync:
						$gallery->set( 'tab', $tab->getInt( 'tabid', 0 ) );

						$gallery->parse( $tab->params, 'gallery_', false );
					}
				}
			}

			$_PLUGINS->trigger( 'gallery_onGalleryAccess', array( &$gallery, &$access ) );

			if ( ! $access )  {
				$galleries[$profileId][$asset][$userId]	=	null;

				return null;
			}

			if ( ! $gallery->id() ) {
				$gallery->cache();
			}

			$galleries[$profileId][$asset][$userId]	=	$gallery;
		}

		return $galleries[$profileId][$asset][$userId];
	}

	/**
	 * @param null|string|array  $assets
	 * @param null|int|UserTable $user
	 * @param array              $defaults
	 * @param Activity|Comments  $stream
	 */
	public function extendParameters( &$assets, &$user, &$defaults, &$stream )
	{
		if ( $stream instanceof Comments ) {
			$defaults['gallery']						=	false;
			$defaults['replies_gallery']				=	false;
		} else {
			$defaults['gallery']						=	true;
			$defaults['comments_gallery']				=	false;
			$defaults['comments_replies_gallery']		=	false;
		}
	}

	/**
	 * @param string                 $output
	 * @param array                  $select
	 * @param array                  $join
	 * @param array                  $where
	 * @param Activity|Notifications $stream
	 */
	public function activityQuery( $output, &$select, &$join, &$where, &$stream )
	{
		global $_CB_database;

		if ( ! $this->isCompatible() ) {
			return;
		}

		$media			=	"SELECT COUNT(*)"
						.	" FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_gallery_items' ) . " AS gallery_item"
						.	" LEFT JOIN " . $_CB_database->NameQuote( '#__comprofiler_plugin_gallery_folders' ) . " AS gallery_folder"
						.	" ON gallery_item." . $_CB_database->NameQuote( 'folder' ) . " = gallery_folder." . $_CB_database->NameQuote( 'id' )
						.	" WHERE gallery_item." . $_CB_database->NameQuote( 'id' ) . " = SUBSTRING_INDEX( SUBSTRING_INDEX( a." . $_CB_database->NameQuote( 'asset' ) . ", '.', 3 ) , '.' , -1 )";

		if ( ! Application::MyUser()->isGlobalModerator() ) {
			if ( Application::MyUser()->getUserId() ) {
				$media	.=	" AND ( gallery_item." . $_CB_database->NameQuote( 'published' ) . " = 1"
						.	" OR gallery_item." . $_CB_database->NameQuote( 'user_id' ) . " = " . Application::MyUser()->getUserId() . " )"
						.	" AND ( gallery_item." . $_CB_database->NameQuote( 'folder' ) . " = 0"
						.	" OR ( gallery_folder." . $_CB_database->NameQuote( 'id' ) . " IS NOT NULL"
						.	" AND ( gallery_folder." . $_CB_database->NameQuote( 'published' ) . " = 1"
						.	" OR gallery_folder." . $_CB_database->NameQuote( 'user_id' ) . ' = ' . Application::MyUser()->getUserId() . " ) ) )";
			} else {
				$media	.=	" AND gallery_item." . $_CB_database->NameQuote( 'published' ) . " = 1"
						.	" AND ( gallery_item." . $_CB_database->NameQuote( 'folder' ) . " = 0"
						.	" OR ( gallery_folder." . $_CB_database->NameQuote( 'id' ) . " IS NOT NULL"
						.	" AND gallery_folder." . $_CB_database->NameQuote( 'published' ) . " = 1 ) )";
			}
		} else {
			$media		.=	" AND ( gallery_item." . $_CB_database->NameQuote( 'folder' ) . " = 0"
						.	" OR gallery_folder." . $_CB_database->NameQuote( 'id' ) . " IS NOT NULL )";
		}

		$where[]		=	"( ( a." . $_CB_database->NameQuote( 'asset' ) . " LIKE " . $_CB_database->Quote( 'gallery.%' )
						.	" AND ( " . $media . " ) > 0 )"
						.	" OR ( a." . $_CB_database->NameQuote( 'asset' ) . " NOT LIKE " . $_CB_database->Quote( 'gallery.%' ) . " ) )";
	}

	/**
	 * @param ActivityTable[]|NotificationTable[] $rows
	 * @param Activity|Notifications              $stream
	 */
	public function activityLoad( &$rows, $stream )
	{
		if ( ! $this->isCompatible() ) {
			return;
		}

		$notification										=	( $stream instanceof NotificationsInterface );
		$items												=	array();

		foreach ( $rows as $k => $row ) {
			if ( ! isset( $rows[$k] ) ) {
				continue;
			}

			$profileId										=	$row->getInt( 'user_id', 0 );

			if ( ! $notification ) {
				$uploads									=	cbToArrayOfInt( $row->params()->getRaw( 'gallery', array() ) );

				if ( $uploads && ( ! $row->getString( 'message' ) ) ) {
					$gallery								=	$this->activityGallery( $profileId );

					if ( ! $gallery ) {
						unset( $rows[$k] );
						continue;
					}

					if ( ! $gallery->reset()->setId( $uploads )->setPublished( ( CBGallery::canModerate( $gallery ) ? null : 1 ) )->items( 'count' ) ) {
						unset( $rows[$k] );
						continue;
					}
				}
			}

			if ( ! preg_match( '/^(?:(.+)\.)?gallery\.(photos|files|videos|music)\.(\d+)(?:\.(like|comment|tag))?/', $row->getString( 'asset' ), $matches ) ) {
				continue;
			}

			$id												=	(int) $matches[3];

			if ( ! $id ) {
				unset( $rows[$k] );
				continue;
			}

			$galleryExists									=	false;

			if ( $notification ) {
				$notifId									=	$row->getInt( 'user', 0 );
				$notifAsset									=	( $matches[1] ? $matches[1] : 'profile.' . $notifId . ',profile.' . $notifId . '.%' );

				if ( ( $notifId !== $profileId )
					 && $this->activityGallery( $notifId, $notifAsset )
				) {
					$items[$notifAsset][$notifId][$k]		=	$id;

					$galleryExists							=	true;
				}
			}

			$profileAsset									=	( $matches[1] ? $matches[1] : 'profile.' . $profileId . ',profile.' . $profileId . '.%' );

			if ( ( ! isset( $items[$profileAsset][$profileId][$k] ) )
				 && $this->activityGallery( $profileId, $profileAsset )
			) {
				$items[$profileAsset][$profileId][$k]		=	$id;

				$galleryExists								=	true;
			}

			if ( ! $galleryExists ) {
				unset( $rows[$k] );
			}
		}

		static $previous									=	array();

		$found												=	array();

		foreach ( $items as $asset => $assetItems ) {
			foreach ( $assetItems as $profileId => $media ) {
				$mediaIds									=	cbToArrayOfInt( array_unique( $media ) );

				if ( ! $mediaIds ) {
					continue;
				}

				$gallery									=	$this->activityGallery( $profileId, $asset );

				if ( ! $gallery ) {
					continue;
				}

				if ( ! isset( $found[$asset][$profileId] ) ) {
					$found[$asset][$profileId]				=	array();
				}

				$ids										=	array_values( $mediaIds );

				foreach ( $mediaIds as $rowId => $itemId ) {
					if ( ! isset( $rows[$rowId] ) ) {
						continue;
					}

					$grouped								=	$rows[$rowId]->params()->getRaw( 'assets', array() );

					foreach ( $grouped as $groupedAsset ) {
						if ( ! preg_match( '/^(?:(.+)\.)?gallery\.(photos|files|videos|music)\.(\d+)(?:\.(like|comment|tag))?/', $groupedAsset, $groupedMatches ) ) {
							continue;
						}

						$ids[]								=	(int) $groupedMatches[3];
					}
				}

				$publishedState								=	( CBGallery::canModerate( $gallery ) ? null : 1 );
				$galleryItems								=	$gallery->reset()->setId( $ids )->setPublished( $publishedState )->items();

				// Check for field assets then check field access since profile field galleries don't store field id with their activity asset:
				foreach ( $galleryItems as $galleryItemId => $galleryItem ) {
					if ( ! preg_match( '/^profile\.(\d+)\.field\.(\d+)/', $galleryItem->getString( 'asset' ), $galleryItemMatches ) ) {
						continue;
					}

					$galleryItemProfileId					=	( isset( $galleryItemMatches[1] ) ? (int) $galleryItemMatches[1] : $gallery->user()->getInt( 'id', 0 ) );
					$galleryItemFieldId						=	( isset( $galleryItemMatches[2] ) ? (int) $galleryItemMatches[2] : $gallery->getInt( 'field', 0 ) );

					if ( ! CBGallery::getField( $galleryItemFieldId, $galleryItemProfileId ) ) {
						unset( $galleryItems[$galleryItemId] );
					}
				}

				$found[$asset][$profileId]					=	( $found[$asset][$profileId] + $galleryItems );
				$folders									=	array();

				foreach ( $mediaIds as $rowId => $itemId ) {
					if ( ( ! array_key_exists( $itemId, $found[$asset][$profileId] ) ) || ( ! isset( $rows[$rowId] ) ) ) {
						continue;
					}

					/** @var ItemTable $currentItem */
					$currentItem							=	$found[$asset][$profileId][$itemId];

					$folder									=	$currentItem->getInt( 'folder' );

					if ( ! $notification ) {
						if ( count( $galleryItems ) > 1 ) {
							if ( $folder ) {
								$overrideComments			=	'gallery.folder.' . $folder;
								$overrideOther				=	$overrideComments;

								if ( preg_match( '/^groupjive\.group\.(\d+)(?:\.([a-zA-Z_]+))?/', $rows[$rowId]->getString( 'asset' ), $groupMatches ) ) {
									$overrideComments		=	'groupjive.group.' . $groupMatches[1] . '.gallery.folder.' . $folder
															.	',gallery.folder.' . $folder;

									$overrideOther			=	'groupjive.group.' . $groupMatches[1] . '.gallery.folder.' . $folder;
								}

								$rows[$rowId]->params()->set( 'overrides.group', 'asset' );
								$rows[$rowId]->params()->set( 'overrides.tags_asset', $overrideOther );
								$rows[$rowId]->params()->set( 'overrides.likes_asset', $overrideOther );
								$rows[$rowId]->params()->set( 'overrides.comments_asset', $overrideComments );
							}
						} else {
							if ( preg_match( '/^groupjive\.group\.(\d+)(?:\.([a-zA-Z_]+))?/', $rows[$rowId]->getString( 'asset' ) ) ) {
								$override					=	$rows[$rowId]->getString( 'asset' )
															.	',gallery.' . $currentItem->getString( 'type' ) . '.' . $itemId;
							} else {
								$override					=	'asset';
							}

							$rows[$rowId]->params()->set( 'overrides.group', 'asset' );
							$rows[$rowId]->params()->set( 'overrides.tags_asset', 'asset' );
							$rows[$rowId]->params()->set( 'overrides.likes_asset', 'asset' );
							$rows[$rowId]->params()->set( 'overrides.comments_asset', $override );
						}
					}

					$folders[$folder][$rowId]				=	$itemId;
				}

				if ( $folders ) {
					$galleryFolders							=	array_keys( $gallery->reset()->setId( array_keys( $folders ) )->setPublished( $publishedState )->folders() );

					foreach ( $folders as $folderId => $folderItems ) {
						if ( count( $folderItems ) <= 1 ) {
							if ( $folderId && ( ! in_array( $folderId, $galleryFolders, true ) ) ) {
								foreach ( $folderItems as $rowId => $itemId ) {
									unset( $found[$asset][$profileId][$itemId] );
								}

								continue;
							}

							continue;
						}

						foreach ( $folderItems as $rowId => $itemId ) {
							if ( ! isset( $rows[$rowId] ) ) {
								continue;
							}

							if ( $folderId ) {
								if ( ! in_array( $folderId, $galleryFolders, true ) ) {
									unset( $found[$asset][$profileId][$itemId] );
									continue;
								}
							}

							if ( ! isset( $previous[$asset][$profileId][$folderId] ) ) {
								$previous[$asset][$profileId][$folderId]		=	array( &$rows[$rowId], &$found[$asset][$profileId][$itemId] );
							} else {
								/** @var ActivityTable $previousRow */
								$previousRow				=	$previous[$asset][$profileId][$folderId][0];
								/** @var ItemTable $previousItem */
								$previousItem				=	$previous[$asset][$profileId][$folderId][1];
								/** @var ItemTable $currentItem */
								$currentItem				=	$found[$asset][$profileId][$itemId];

								/** @var array $previousItems */
								$previousItems				=	$previousRow->params()->getRaw( 'overrides.gallery_items', array() );
								$dateDiff					=	Application::Date( $previousItem->getString( 'date' ), 'UTC' )->diff( $currentItem->getString( 'date' ) );

								if ( ( $dateDiff->days === 0 ) && ( $dateDiff->h === 0 ) && ( $dateDiff->i <= 15 ) ) {
									$previousItems[]		=	$itemId;

									$previousRow->params()->set( 'overrides.gallery_items', $previousItems );

									unset( $rows[$rowId] );
								} else {
									$previous[$asset][$profileId][$folderId]	=	array( &$rows[$rowId], &$found[$asset][$profileId][$itemId] );
								}
							}
						}
					}
				}
			}
		}

		$isFound											=	array();
		$notFound											=	array();

		foreach ( $items as $asset => $assetItems ) {
			foreach ( $assetItems as $profileId => $media ) {
				$mediaIds									=	cbToArrayOfInt( array_unique( $media ) );

				if ( ! $mediaIds ) {
					continue;
				}

				if ( ! isset( $found[$asset][$profileId] ) ) {
					$found[$asset][$profileId]				=	array();
				}

				foreach ( $mediaIds as $rowId => $itemId ) {
					if ( in_array( $rowId, $isFound, true ) ) {
						continue;
					}

					if ( array_key_exists( $itemId, $found[$asset][$profileId] ) ) {
						$isFound[$rowId]					=	$rowId;

						if ( in_array( $rowId, $notFound, true ) ) {
							unset( $notFound[$rowId] );
						}
					} else {
						$notFound[$rowId]					=	$rowId;
					}
				}
			}
		}

		foreach ( $notFound as $rowId ) {
			if ( ! isset( $rows[$rowId] ) ) {
				continue;
			}

			unset( $rows[$rowId] );
		}
	}

	/**
	 * @param ActivityTable|NotificationTable $row
	 * @param null|string                     $title
	 * @param null|string                     $date
	 * @param null|string                     $message
	 * @param null|string                     $insert
	 * @param null|string                     $footer
	 * @param array                           $menu
	 * @param Activity|Notifications          $stream
	 * @param string                          $output
	 */
	public function activityDisplay( &$row, &$title, &$date, &$message, &$insert, &$footer, &$menu, $stream, $output )
	{
		if ( ! $this->isCompatible() ) {
			return;
		}

		$uploads				=	cbToArrayOfInt( $row->params()->getRaw( 'gallery', array() ) );
		$matches				=	array();

		if ( ( ! $uploads ) && ( ! preg_match( '/^(?:(.+)\.)?gallery\.(photos|files|videos|music)\.(\d+)(?:\.(like|comment|tag))?/', $row->getString( 'asset' ), $matches ) ) ) {
			return;
		}

		/** @var array $ids */
		$ids					=	$row->params()->getRaw( 'overrides.gallery_items', array() );
		$mediaId				=	( isset( $matches[3] ) ? (int) $matches[3] : 0 );

		if ( $mediaId ) {
			array_unshift( $ids, $mediaId );
		}

		$grouped				=	$row->params()->getRaw( 'assets', array() );

		foreach ( $grouped as $groupedAsset ) {
			if ( ! preg_match( '/^(?:(.+)\.)?gallery\.(photos|files|videos|music)\.(\d+)(?:\.(like|comment|tag))?/', $groupedAsset, $groupedMatches ) ) {
				continue;
			}

			$ids[]				=	(int) $groupedMatches[3];
		}

		$ids					=	array_unique( array_merge( $ids, $uploads ) );

		if ( ! $ids ) {
			return;
		}

		$notification			=	( $stream instanceof NotificationsInterface );
		$galleryAsset			=	( isset( $matches[1] ) ? $matches[1] : null );
		$notifId				=	$row->getInt( 'user', 0 );
		$profileId				=	$row->getInt( 'user_id', 0 );
		$items					=	array();

		if ( $notification && ( $notifId !== $profileId ) ) {
			$gallery			=	$this->activityGallery( $notifId, $galleryAsset );

			if ( $gallery ) {
				$items			=	$gallery->reset()->setId( $ids )->setPublished( ( CBGallery::canModerate( $gallery ) ? null : 1 ) )->items();
			}

			if ( ! $items ) {
				$gallery		=	$this->activityGallery( $profileId, $galleryAsset );

				if ( $gallery ) {
					$items		=	$gallery->reset()->setId( $ids )->setPublished( ( CBGallery::canModerate( $gallery ) ? null : 1 ) )->items();
				}
			}
		} else {
			$gallery			=	$this->activityGallery( $profileId, $galleryAsset );

			if ( $gallery ) {
				$items			=	$gallery->reset()->setId( $ids )->setPublished( ( CBGallery::canModerate( $gallery ) ? null : 1 ) )->items();
			}
		}

		if ( ! $items ) {
			return;
		}

		CBGallery::getTemplate( array( 'activity', 'item_container', 'embed' ) );

		if ( $matches && ( ! $notification ) ) {
			if ( count( $items ) > 1 ) {
				$folderId			=	null;

				/** @var ItemTable[] $items */
				foreach ( $items as $i => $item ) {
					if ( $folderId !== null ) {
						break;
					}

					$folderId		=	$item->getInt( 'folder', 0 );
				}

				if ( $folderId ) {
					$overrideComments		=	'gallery.folder.' . $folderId;
					$overrideOther			=	$overrideComments;

					if ( preg_match( '/^groupjive\.group\.(\d+)(?:\.([a-zA-Z_]+))?/', $row->getString( 'asset' ), $groupMatches ) ) {
						$overrideComments	=	'groupjive.group.' . $groupMatches[1] . '.gallery.folder.' . $folderId
											.	',gallery.folder.' . $folderId;

						$overrideOther		=	'groupjive.group.' . $groupMatches[1] . '.gallery.folder.' . $folderId;
					}

					$row->params()->set( 'overrides.tags_asset', $overrideOther );
					$row->params()->set( 'overrides.likes_asset', $overrideOther );
					$row->params()->set( 'overrides.comments_asset', $overrideComments );
				}
			} else {
				if ( preg_match( '/^groupjive\.group\.(\d+)(?:\.([a-zA-Z_]+))?/', $row->getString( 'asset' ) ) ) {
					$override		=	$row->getString( 'asset' )
									.	',gallery.' . $matches[2] . '.' . $mediaId;
				} else {
					$override		=	'asset';
				}

				$row->params()->set( 'overrides.tags_asset', 'asset' );
				$row->params()->set( 'overrides.likes_asset', 'asset' );
				$row->params()->set( 'overrides.comments_asset', $override );
			}
		}

		\HTML_cbgalleryActivity::showActivity( $row, $title, $date, $message, $insert, $footer, $menu, $stream, $matches, $items, $gallery, $this, $output );
	}

	/**
	 * @param CommentTable $row
	 * @param null|string  $message
	 * @param null|string  $insert
	 * @param null|string  $date
	 * @param null|string  $footer
	 * @param array        $menu
	 * @param Comments     $stream
	 * @param string       $output
	 */
	public function commentDisplay( &$row, &$message, &$insert, &$date, &$footer, &$menu, $stream, $output )
	{
		if ( ! $this->isCompatible() ) {
			return;
		}

		$uploads		=	cbToArrayOfInt( $row->params()->getRaw( 'gallery', array() ) );

		if ( ! $uploads ) {
			return;
		}

		$gallery		=	$this->activityGallery( $row->getInt( 'user_id', 0 ) );

		if ( ! $gallery ) {
			return;
		}

		$items			=	$gallery->reset()->setId( $uploads )->setPublished( ( CBGallery::canModerate( $gallery ) ? null : 1 ) )->items();

		if ( ! $items ) {
			return;
		}

		$title			=	null;
		$matches		=	array();

		CBGallery::getTemplate( array( 'activity', 'item_container', 'embed' ) );

		\HTML_cbgalleryActivity::showActivity( $row, $title, $date, $message, $insert, $footer, $menu, $stream, $matches, $items, $gallery, $this, $output );
	}

	/**
	 * @param string $asset
	 * @param mixed  $source
	 */
	public function assetSource( $asset, &$source )
	{
		if ( ! preg_match( '/^(?:(.+)\.)?gallery\.(photos|files|videos|music|folder)\.(\d+)(?:\.(like|comment|tag))?/', $asset, $matches ) ) {
			return;
		}

		if ( ( isset( $matches[2] ) ? $matches[2] : null ) === 'folder' ) {
			$row	=	new FolderTable();
		} else {
			$row	=	new ItemTable();
		}

		$row->load( ( isset( $matches[3] ) ? (int) $matches[3] : 0 ) );

		if ( ! $row->getInt( 'id', 0 ) ) {
			return;
		}

		$source		=	$row;
	}

	/**
	 * @param array             $buttons
	 * @param UserTable         $viewer
	 * @param Activity|Comments $stream
	 * @param string            $output
	 * @return mixed
	 */
	public function activityUploadNew( &$buttons, $viewer, $stream, $output )
	{
		if ( ! $this->isCompatible() || ( ! $stream->getBool( 'gallery', false ) ) ) {
			return null;
		}

		$gallery	=	$this->activityGallery( $viewer->getInt( 'id', 0 ) );

		if ( ! $gallery ) {
			return null;
		}

		CBGallery::getTemplate( 'activity_new' );

		return \HTML_cbgalleryActivityNew::showActivityNew( $buttons, $viewer, $stream, $gallery, $this, $output );
	}

	/**
	 * @param ActivityTable|CommentTable $row
	 * @param array                      $buttons
	 * @param UserTable                  $viewer
	 * @param Activity|Comments          $stream
	 * @param string                     $output
	 * @return mixed
	 */
	public function activityUploadEdit( &$row, &$buttons, $viewer, $stream, $output )
	{
		if ( ! $this->isCompatible() ) {
			return null;
		}

		$uploads	=	cbToArrayOfInt( $row->params()->getRaw( 'gallery', array() ) );

		if ( ! $uploads ) {
			return null;
		}

		$gallery	=	$this->activityGallery( $viewer->getInt( 'id', 0 ) );

		if ( ! $gallery ) {
			return null;
		}

		$items		=	$gallery->reset()->setId( $uploads )->items();

		if ( ! $items ) {
			return null;
		}

		CBGallery::getTemplate( array( 'activity_edit', 'item_edit_micro', 'item_container' ) );

		return \HTML_cbgalleryActivityEdit::showActivityEdit( $row, $items, $buttons, $viewer, $stream, $gallery, $this, $output );
	}

	/**
	 * @param Activity|Comments          $stream
	 * @param mixed                      $source
	 * @param ActivityTable|CommentTable $row
	 */
	public function activityUploadSave( $stream, $source, &$row )
	{
		if ( ( ! $this->isCompatible() ) || ( $row->getInt( 'user_id', 0 ) !== Application::MyUser()->getUserId() ) || ( ! $stream->getBool( 'gallery', false ) ) ) {
			return;
		}

		$gallery			=	$this->activityGallery( $row->getInt( 'user_id', 0 ) );

		if ( ! $gallery ) {
			return;
		}

		$items				=	cbToArrayOfInt( Application::Input()->getRaw( 'items', array() ) );

		if ( ! $items ) {
			return;
		}

		$media				=	array();

		foreach ( $gallery->reset()->setId( $items )->items() as $item ) {
			$media[]		=	$item->getInt( 'id', 0 );
		}

		if ( $media ) {
			$row->params()->set( 'overrides.message', false );
			$row->params()->set( 'gallery', $media );
		}
	}
}