<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\Gallery\Table;

use CBLib\Application\Application;
use CBLib\Database\Table\Table;
use CBLib\Language\CBTxt;
use CBLib\Registry\Registry;
use CB\Plugin\Gallery\Gallery;
use CB\Plugin\Gallery\CBGallery;
use CB\Database\Table\UserTable;

defined('CBLIB') or die();

class FolderTable extends Table
{
	/** @var int  */
	public $id				=	null;
	/** @var int  */
	public $user_id			=	null;
	/** @var string  */
	public $asset			=	null;
	/** @var string  */
	public $title			=	null;
	/** @var string  */
	public $description		=	null;
	/** @var int  */
	public $thumbnail		=	null;
	/** @var string  */
	public $date			=	null;
	/** @var int  */
	public $published		=	null;
	/** @var string  */
	public $params			=	null;

	/** @var Registry  */
	protected $_params		=	null;

	/**
	 * Table name in database
	 * @var string
	 */
	protected $_tbl			=	'#__comprofiler_plugin_gallery_folders';

	/**
	 * Primary key(s) of table
	 * @var string
	 */
	protected $_tbl_key		=	'id';

	/**
	 * @return bool
	 */
	public function check()
	{
		if ( ! $this->getInt( 'user_id', 0 ) ) {
			$this->setError( CBTxt::T( 'Owner not specified!' ) );

			return false;
		}

		return true;
	}

	/**
	 * @param bool $updateNulls
	 * @return bool
	 */
	public function store( $updateNulls = false )
	{
		global $_CB_database, $_PLUGINS;

		$new			=	( ! $this->getInt( 'id', 0 ) );
		$old			=	new self();

		if ( ! $this->getString( 'asset' ) ) {
			$this->set( 'asset', 'profile.' . $this->getInt( 'user_id', 0 ) );
		}

		$this->set( 'published', $this->getInt( 'published', 0 ) );
		$this->set( 'date', $this->getString( 'date', Application::Database()->getUtcDateTime() ) );

		if ( ! $new ) {
			$old->load( $this->getInt( 'id', 0 ) );

			$_PLUGINS->trigger( 'gallery_onBeforeUpdateFolder', array( &$this, $old ) );
		} else {
			$_PLUGINS->trigger( 'gallery_onBeforeCreateFolder', array( &$this ) );
		}

		if ( ! parent::store( $updateNulls ) ) {
			return false;
		}

		if ( $old->getInt( 'id', 0 ) && ( $old->getInt( 'user_id', 0 ) !== $this->getInt( 'user_id', 0 ) ) ) {
			$query		=	'SELECT *'
						.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_gallery_items' )
						.	"\n WHERE " . $_CB_database->NameQuote( 'folder' ) . " = " . $this->getInt( 'id', 0 )
						.	"\n AND " . $_CB_database->NameQuote( 'user_id' ) . " !== " . $this->getInt( 'user_id', 0 );
			$_CB_database->setQuery( $query );
			$items		=	$_CB_database->loadObjectList( null, '\CB\Plugin\Gallery\Table\ItemTable', array( $_CB_database ) );

			/** @var ItemTable[] $items */
			foreach ( $items as $item ) {
				$item->set( 'user_id', $this->getInt( 'user_id', 0 ) );

				$item->store();
			}
		}

		if ( ! $new ) {
			$_PLUGINS->trigger( 'gallery_onAfterUpdateFolder', array( $this, $old ) );
		} else {
			$_PLUGINS->trigger( 'gallery_onAfterCreateFolder', array( $this ) );
		}

		return true;
	}

	/**
	 * @param null|int $id
	 * @return bool
	 */
	public function delete( $id = null )
	{
		global $_CB_database, $_PLUGINS;

		$_PLUGINS->trigger( 'gallery_onBeforeDeleteFolder', array( &$this ) );

		if ( ! parent::delete( $id ) ) {
			return false;
		}

		if ( $this->getInt( 'id', 0 ) ) {
			$query		=	'SELECT *'
						.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_gallery_items' )
						.	"\n WHERE " . $_CB_database->NameQuote( 'folder' ) . " = " . $this->getInt( 'id', 0 );
			$_CB_database->setQuery( $query );
			$items		=	$_CB_database->loadObjectList( null, '\CB\Plugin\Gallery\Table\ItemTable', array( $_CB_database ) );

			/** @var ItemTable[] $items */
			foreach ( $items as $item ) {
				$item->delete();
			}
		}

		$_PLUGINS->trigger( 'gallery_onAfterDeleteFolder', array( $this ) );

		return true;
	}

	/**
	 * @return Registry
	 */
	public function params()
	{
		if ( ! ( $this->getRaw( '_params' ) instanceof Registry ) ) {
			$this->set( '_params', new Registry( $this->getRaw( 'params' ) ) );
		}

		return $this->getRaw( '_params' );
	}

	/**
	 * @return UserTable|null
	 */
	public function source()
	{
		global $_PLUGINS;

		static $cache		=	array();

		$id					=	$this->getString( 'asset' );

		if ( ! isset( $cache[$id] ) ) {
			$source			=	CBGallery::getSource( $id );

			$_PLUGINS->trigger( 'gallery_onFolderSource', array( $this, &$source ) );

			$cache[$id]		=	$source;
		}

		return $cache[$id];
	}

	/**
	 * Returns the items in this folder
	 *
	 * @param bool         $count
	 * @param null|Gallery $gallery
	 * @return int|ItemTable[]
	 */
	public function items( $count = false, $gallery = null )
	{
		global $_CB_database;

		if ( $count && ( $this->getRaw( '_items' ) !== null ) ) {
			return $this->getInt( '_items', 0 );
		}

		$id									=	$this->getInt( 'id', 0 );

		if ( ! $id ) {
			return ( $count ? 0 : array() );
		}

		if ( $gallery ) {
			return $gallery->reset()->setFolder( $id )->setPublished( $gallery->getRaw( 'published' ) )->items( $count );
		}

		static $cache						=	array();

		$userId								=	Application::MyUser()->getUserId();

		if ( ! isset( $cache[$id][$userId][$count] ) ) {
			if ( $count ) {
				$query						=	'SELECT COUNT(*)'
											.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_gallery_items' )
											.	"\n WHERE " . $_CB_database->NameQuote( 'folder' ) . " = " . $id
											.	( ( ( $userId !== $this->getInt( 'user_id', 0 ) ) && ( ! Application::User( $userId )->isGlobalModerator() ) ) ? "\n AND " . $_CB_database->NameQuote( 'published' ) . " = 1" : null );
				$_CB_database->setQuery( $query );
				$rows						=	(int) $_CB_database->loadResult();
			} else {
				$query						=	'SELECT *'
											.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_gallery_items' )
											.	"\n WHERE " . $_CB_database->NameQuote( 'folder' ) . " = " . $id
											.	( ( ( $userId !== $this->getInt( 'user_id', 0 ) ) && ( ! Application::User( $userId )->isGlobalModerator() ) ) ? "\n AND " . $_CB_database->NameQuote( 'published' ) . " = 1" : null );
				$_CB_database->setQuery( $query );
				$rows						=	$_CB_database->loadObjectList( 'id', '\CB\Plugin\Gallery\Table\ItemTable', array( $_CB_database ) );
			}

			$cache[$id][$userId][$count]	=	$rows;
		}

		return $cache[$id][$userId][$count];
	}

	/**
	 * Returns folders thumbnail
	 *
	 * @param Gallery  $gallery
	 * @param int|null $tabIndex
	 * @return string
	 */
	public function thumbnail( $gallery = null, $tabIndex = null )
	{
		$thumbnail		=	$this->getInt( 'thumbnail', 0 );

		if ( $gallery && $thumbnail ) {
			$item		=	$gallery->item( $thumbnail );

			if ( $item->getInt( 'id', 0 ) && ( $item->getInt( 'folder', 0 ) === $this->getInt( 'id', 0 ) ) && ( ( $item->getInt( 'published', 1 ) ) || ( ( Application::MyUser()->getUserId() === $item->getInt( 'user_id', 0 ) ) || CBGallery::canModerate( $gallery ) ) ) ) {
				return $item->thumbnail( $gallery, $tabIndex );
			}
		}

		return '<span class="display-1 align-middle galleryImageIcon"><span class="fa fa-folder-open-o"></span></span>';
	}
}