<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\Gallery;

use CB\Plugin\Gallery\Table\FolderTable;
use CBLib\Application\Application;
use CBLib\Input\Get;
use CBLib\Registry\ParamsInterface;
use CBLib\Registry\Registry;
use CB\Database\Table\TabTable;
use CB\Database\Table\FieldTable;
use CBLib\Language\CBTxt;
use CBLib\Registry\GetterInterface;
use GuzzleHttp\Client;
use CB\Plugin\Gallery\Table\ItemTable;
use CB\Database\Table\UserTable;
use Exception;

defined('CBLIB') or die();

class CBGallery
{

	/**
	 * @return Registry
	 */
	public static function getGlobalParams()
	{
		global $_PLUGINS;

		static $params	=	null;

		if ( ! $params ) {
			$plugin		=	$_PLUGINS->getLoadedPlugin( 'user', 'cbgallery' );
			$params		=	new Registry();

			if ( $plugin ) {
				$params->load( $plugin->params );
			}
		}

		return $params;
	}

	/**
	 * Try to find the gallery asset from item or folder id
	 *
	 * @param string $type
	 * @param int    $id
	 * @return null|string
	 */
	public static function getAsset( $type, $id )
	{
		if ( ! $id ) {
			return null;
		}

		static $cache				=	array();

		if ( ! isset( $cache[$type][$id] ) ) {
			$asset					=	null;

			switch ( $type ) {
				case 'folder':
					$row			=	new FolderTable();

					$row->load( $id );

					$asset			=	$row->getString( 'asset' );
					break;
				case 'item':
					$row			=	new ItemTable();

					$row->load( $id );

					$asset			=	$row->getString( 'asset' );
					break;
			}

			$cache[$type][$id]		=	$asset;
		}

		return $cache[$type][$id];
	}

	/**
	 * Try to build the asset source
	 *
	 * @param string $asset
	 * @return mixed
	 */
	public static function getSource( $asset )
	{
		global $_PLUGINS;

		if ( ! $asset ) {
			return null;
		}

		static $cache				=	array();

		if ( ! isset( $cache[$asset] ) ) {
			$source					=	null;

			if ( preg_match( '/^profile\.(\d+)/', $asset, $matches ) ) {
				$source				=	\CBuser::getInstance( (int) $matches[1] )->getUserData();
			}

			$_PLUGINS->trigger( 'gallery_onAssetSource', array( $asset, &$source ) );

			$cache[$asset]			=	$source;
		}

		return $cache[$asset];
	}

	/**
	 * Utility function for grabbing a field while also ensuring proper display access to it
	 *
	 * @param int      $fieldId
	 * @param int|null $profileId
	 * @return FieldTable|null
	 */
	public static function getField( $fieldId, $profileId = null )
	{
		if ( ! $fieldId ) {
			return null;
		}

		$userId								=	Application::MyUser()->getUserId();

		if ( ! $profileId ) {
			$profileId						=	$userId;
		}

		static $fields						=	array();

		if ( ! isset( $fields[$profileId][$userId] ) ) {
			$profileUser					=	\CBuser::getInstance( $profileId, false );

			$fields[$profileId][$userId]	=	$profileUser->_getCbTabs( false )->_getTabFieldsDb( null, $profileUser->getUserData(), 'profile' );
		}

		if ( ! isset( $fields[$profileId][$userId][$fieldId] ) ) {
			return null;
		}

		$field								=	$fields[$profileId][$userId][$fieldId];

		if ( ! ( $field->params instanceof ParamsInterface ) ) {
			$field->params					=	new Registry( $field->params );
		}

		return $field;
	}

	/**
	 * Utility function for grabbing the gallery tab while also ensuring proper display access to it
	 *
	 * @param int      $tabId
	 * @param int|null $profileId
	 * @return TabTable|null
	 */
	public static function getTab( $tabId, $profileId = null )
	{
		static $profileTab					=	null;

		if ( ! $tabId ) {
			if ( $profileTab === null ) {
				$profileTab					=	new TabTable();

				$profileTab->load( array( 'pluginclass' => 'cbgalleryTab' ) );
			}

			$tabId							=	$profileTab->getInt( 'tabid', 0 );
		}

		if ( ! $tabId ) {
			return null;
		}

		$userId								=	Application::MyUser()->getUserId();

		if ( ! $profileId ) {
			$profileId						=	$userId;
		}

		static $tabs						=	array();

		if ( ! isset( $tabs[$profileId][$userId] ) ) {
			$profileUser					=	\CBuser::getInstance( $profileId, false );

			$tabs[$profileId][$userId]		=	$profileUser->_getCbTabs( false )->_getTabsDb( $profileUser->getUserData(), 'profile', false );
		}

		if ( ! isset( $tabs[$profileId][$userId][$tabId] ) ) {
			return null;
		}

		$tab								=	$tabs[$profileId][$userId][$tabId];

		if ( ! ( $tab->params instanceof ParamsInterface ) ) {
			$tab->params					=	new Registry( $tab->params );
		}

		return $tab;
	}

	/**
	 * Returns an array of users connections
	 *
	 * @param int $profileId
	 * @return array
	 */
	public static function getConnections( $profileId )
	{
		if ( ! $profileId ) {
			return array();
		}

		static $cache				=	array();

		if ( ! isset( $cache[$profileId] ) ) {
			$cbConnection			=	new \cbConnection( $profileId );

			$cache[$profileId]		=	$cbConnection->getActiveConnections( $profileId );
		}

		return $cache[$profileId];
	}

	/**
	 * Checks if a user can create folders in the supplied gallery
	 *
	 * @param Gallery        $gallery
	 * @param null|UserTable $user
	 * @return bool
	 */
	public static function canCreateFolders( $gallery, $user = null )
	{
		global $_PLUGINS;

		static $cache								=	array();

		if ( ( ! $gallery->getBool( 'folders', true ) ) || ( ! $gallery->getBool( 'folders_create', true ) ) ) {
			return false;
		}

		if ( ! $user ) {
			$user									=	\CBuser::getMyUserDataInstance();
		}

		$userId										=	$user->getInt( 'id', 0 );

		if ( ! $userId ) {
			return false;
		}

		$galleryId									=	$gallery->id();

		if ( ! isset( $cache[$userId][$galleryId] ) ) {
			if ( self::canModerate( $gallery, $user ) ) {
				$cache[$userId][$galleryId]			=	true;

				return true;
			}

			if ( preg_match( '/^profile(?:\.(\d+)(?:\.field\.(\d+))?)?/', $gallery->asset(), $matches ) ) {
				if ( ( isset( $matches[1] ) ? (int) $matches[1] : 0 ) !== $userId ) {
					$cache[$userId][$galleryId]		=	false;

					return false;
				}

				$profileId							=	( isset( $matches[1] ) ? (int) $matches[1] : $gallery->user()->getInt( 'id', 0 ) );
				$fieldId							=	( isset( $matches[2] ) ? (int) $matches[2] : $gallery->getInt( 'field', 0 ) );
				$tabId								=	$gallery->getInt( 'tab', 0 );

				if ( $fieldId ) {
					$field							=	CBGallery::getField( $fieldId, $profileId );

					if ( ! $field ) {
						$cache[$userId][$galleryId]	=	false;

						return false;
					}
				} else {
					$tab							=	CBGallery::getTab( $tabId, $profileId );

					if ( ! $tab ) {
						$cache[$userId][$galleryId]	=	false;

						return false;
					}
				}
			} elseif ( $gallery->getString( 'content.context' ) ) {
				$authorId							=	$gallery->getInt( 'content.author', 0 );

				if ( $authorId && ( $userId !== $authorId ) ) {
					$cache[$userId][$galleryId]		=	false;

					return false;
				}
			}

			if ( ! Application::User( (int) $userId )->canViewAccessLevel( $gallery->getInt( 'folders_create_access', 2 ) ) ) {
				$cache[$userId][$galleryId]			=	false;

				return false;
			}

			if ( self::createLimitedFolders( $gallery, $user ) ) {
				$cache[$userId][$galleryId]			=	false;

				return false;
			}

			$access									=	true;

			$_PLUGINS->trigger( 'gallery_onGalleryFoldersCreateAccess', array( &$access, $user, $gallery ) );

			$cache[$userId][$galleryId]				=	$access;
		}

		return $cache[$userId][$galleryId];
	}

	/**
	 * Checks if a user is create limited for folders in a gallery
	 *
	 * @param Gallery        $gallery
	 * @param null|UserTable $user
	 * @return bool
	 */
	public static function createLimitedFolders( $gallery, $user = null )
	{
		static $cache						=	array();

		if ( ! $user ) {
			$user							=	\CBuser::getMyUserDataInstance();
		}

		$userId								=	$user->getInt( 'id', 0 );

		if ( ! $userId ) {
			return false;
		}

		$galleryId							=	$gallery->id();

		if ( ! isset( $cache[$userId][$galleryId] ) ) {
			$createLimit					=	$gallery->getString( 'folders_create_limit', 'custom' );

			if ( $createLimit && ( $createLimit !== 'custom' ) ) {
				$limitField					=	\CBuser::getInstance( $userId, false )->getField( $createLimit, null, 'php', 'none', 'profile', 0, true );

				if ( is_array( $limitField ) ) {
					$createLimit			=	array_shift( $limitField );

					if ( is_array( $createLimit ) ) {
						$createLimit		=	implode( '|*|', $createLimit );
					}
				} else {
					$createLimit			=	$user->getInt( $createLimit, 0 );
				}

				$createLimit				=	(int) $createLimit;
			} else {
				$createLimit				=	$gallery->getInt( 'folders_create_limit_custom', 0 );
			}

			if ( $createLimit ) {
				/** @var Gallery $gallery */
				$gallery					=	$gallery->reset()->setUserId( $userId );

				if ( $gallery->folders( 'count' ) >= $createLimit ) {
					$createLimited			=	true;
				} else {
					$createLimited			=	false;
				}
			} else {
				$createLimited				=	false;
			}

			$cache[$userId][$galleryId]		=	$createLimited;
		}

		return $cache[$userId][$galleryId];
	}

	/**
	 * Checks if a user can create items in the supplied gallery
	 *
	 * @param string         $type
	 * @param string         $method
	 * @param Gallery        $gallery
	 * @param null|UserTable $user
	 * @return bool
	 */
	public static function canCreateItems( $type, $method, $gallery, $user = null )
	{
		global $_PLUGINS;

		static $cache													=	array();

		if ( ! $gallery->getBool( 'items_create', true ) ) {
			return false;
		}

		if ( ! $user ) {
			$user														=	\CBuser::getMyUserDataInstance();
		}

		$userId															=	$user->getInt( 'id', 0 );

		if ( ! $userId ) {
			return false;
		}

		$galleryId														=	$gallery->id();

		if ( ! isset( $cache[$userId][$type][$method][$galleryId] ) ) {
			$folderId													=	$gallery->getInt( 'folder', 0 );

			if ( $folderId ) {
				if ( ! $gallery->getBool( 'folders', true ) ) {
					$cache[$userId][$type][$method][$galleryId]			=	false;

					return false;
				}

				$folder													=	$gallery->folder( $folderId );

				if ( ( ! $folder->getInt( 'id', 0 ) )
					 || ( ( ! $folder->getInt( 'published', 1 ) ) && ( $userId !== $folder->getInt( 'user_id', 0 ) ) && ( ! self::canModerate( $gallery ) ) )
					 || ( ( $folder->getInt( 'published', 1 ) === -1 ) && $gallery->getBool( 'folders_create_approval', false ) && ( ! self::canModerate( $gallery ) ) )
				) {
					$cache[$userId][$type][$method][$galleryId]			=	false;

					return false;
				}
			}

			$create														=	false;

			foreach ( $gallery->types() as $galleryType ) {
				if ( ! in_array( $type, array( 'all', $galleryType ), true ) ) {
					continue;
				}

				if ( $create ) {
					break;
				}

				if ( $gallery->getBool( $galleryType . '_create', true ) ) {
					$create												=	true;
				}
			}

			if ( ! $create ) {
				$cache[$userId][$type][$method][$galleryId]				=	false;

				return false;
			}

			if ( self::canModerate( $gallery, $user ) ) {
				$cache[$userId][$type][$method][$galleryId]				=	true;

				return true;
			}

			if ( preg_match( '/^profile(?:\.(\d+)(?:\.field\.(\d+))?)?/', $gallery->asset(), $matches ) ) {
				if ( ( isset( $matches[1] ) ? (int) $matches[1] : 0 ) !== $userId ) {
					$cache[$userId][$type][$method][$galleryId]			=	false;

					return false;
				}

				$profileId												=	( isset( $matches[1] ) ? (int) $matches[1] : $gallery->user()->getInt( 'id', 0 ) );
				$fieldId												=	( isset( $matches[2] ) ? (int) $matches[2] : $gallery->getInt( 'field', 0 ) );
				$tabId													=	$gallery->getInt( 'tab', 0 );

				if ( $fieldId ) {
					$field												=	CBGallery::getField( $fieldId, $profileId );

					if ( ! $field ) {
						$cache[$userId][$galleryId]						=	false;

						return false;
					}
				} else {
					$tab												=	CBGallery::getTab( $tabId, $profileId );

					if ( ! $tab ) {
						$cache[$userId][$galleryId]						=	false;

						return false;
					}
				}
			} elseif ( $gallery->getString( 'content.context' ) ) {
				$authorId												=	$gallery->getInt( 'content.author', 0 );

				if ( $authorId && ( $userId !== $authorId ) ) {
					$cache[$userId][$galleryId]							=	false;

					return false;
				}
			}

			$access														=	false;

			foreach ( $gallery->types() as $galleryType ) {
				if ( ! in_array( $type, array( 'all', $galleryType ), true ) ) {
					continue;
				}

				if ( $access ) {
					break;
				}

				$upload													=	$gallery->getBool( $galleryType . '_upload', true );
				$link													=	$gallery->getBool( $galleryType . '_link', true );

				if ( ! $gallery->getBool( $galleryType . '_create', true ) ) {
					$access												=	false;
					continue;
				}

				if ( ( $method === 'upload' ) && ( ! $upload ) ) {
					$access												=	false;
					continue;
				}

				if ( ( $method === 'link' ) && ( ! $link ) ) {
					$access												=	false;
					continue;
				}

				if ( ( ! $upload ) && ( ! $link ) ) {
					$access												=	false;
					continue;
				}

				if ( ! Application::User( (int) $userId )->canViewAccessLevel( $gallery->getInt( $galleryType . '_create_access', 2 ) ) ) {
					$access												=	false;
					continue;
				}

				if ( self::createLimitedItems( $type, $gallery, $user ) ) {
					$access												=	false;
					continue;
				}

				$access													=	true;
			}

			if ( ! $access ) {
				$cache[$userId][$type][$method][$galleryId]				=	false;

				return false;
			}

			$_PLUGINS->trigger( 'gallery_onGalleryItemsCreateAccess', array( &$access, $type, $method, $user, $gallery ) );

			$cache[$userId][$type][$method][$galleryId]					=	$access;
		}

		return $cache[$userId][$type][$method][$galleryId];
	}

	/**
	 * Checks if a user is create limited for items in a gallery
	 *
	 * @param string         $type
	 * @param Gallery        $gallery
	 * @param null|UserTable $user
	 * @return bool
	 */
	public static function createLimitedItems( $type, $gallery, $user = null )
	{
		static $cache								=	array();

		if ( ! $user ) {
			$user									=	\CBuser::getMyUserDataInstance();
		}

		$userId										=	$user->getInt( 'id', 0 );

		if ( ( ! $userId ) || ( ! in_array( $type, $gallery->types(), true ) ) ) {
			return false;
		}

		$galleryId									=	$gallery->id();

		if ( ! isset( $cache[$userId][$type][$galleryId] ) ) {
			$createLimit							=	$gallery->getString( $type . '_create_limit', 'custom' );

			if ( $createLimit && ( $createLimit !== 'custom' ) ) {
				$limitField							=	\CBuser::getInstance( $userId, false )->getField( $createLimit, null, 'php', 'none', 'profile', 0, true );

				if ( is_array( $limitField ) ) {
					$createLimit					=	array_shift( $limitField );

					if ( is_array( $createLimit ) ) {
						$createLimit				=	implode( '|*|', $createLimit );
					}
				} else {
					$createLimit					=	$user->getInt( $createLimit, 0 );
				}

				$createLimit						=	(int) $createLimit;
			} else {
				$createLimit						=	$gallery->getInt( $type . '_create_limit_custom', 0 );
			}

			if ( $createLimit ) {
				/** @var Gallery $gallery */
				$gallery							=	$gallery->reset()->setUserId( $userId )->setType( $type );

				if ( $gallery->items( 'count' ) >= $createLimit ) {
					$createLimited					=	true;
				} else {
					$createLimited					=	false;
				}

				$gallery->set( 'type', null );
			} else {
				$createLimited						=	false;
			}

			$cache[$userId][$type][$galleryId]		=	$createLimited;
		}

		return $cache[$userId][$type][$galleryId];
	}

	/**
	 * Checks if a user can moderate the gallery
	 *
	 * @param Gallery        $gallery
	 * @param null|UserTable $user
	 * @return bool
	 */
	public static function canModerate( $gallery, $user = null )
	{
		global $_PLUGINS;

		static $cache						=	array();

		if ( ! $user ) {
			$user							=	\CBuser::getMyUserDataInstance();
		}

		$userId								=	$user->getInt( 'id', 0 );

		if ( ! $userId ) {
			return false;
		}

		if ( Application::User( $userId )->isGlobalModerator() ) {
			return true;
		}

		if ( in_array( $userId, $gallery->getModerators(), true ) ) {
			return true;
		}

		$galleryId							=	$gallery->id();

		if ( ! isset( $cache[$userId][$galleryId] ) ) {
			$access							=	false;

			$_PLUGINS->trigger( 'gallery_onGalleryModerateAccess', array( &$access, $user, $gallery ) );

			$cache[$userId][$galleryId]		=	$access;
		}

		return $cache[$userId][$galleryId];
	}

	/**
	 * @param null|array $files
	 * @param bool       $loadGlobal
	 * @param bool       $loadHeader
	 */
	public static function getTemplate( $files = null, $loadGlobal = true, $loadHeader = true )
	{
		global $_CB_framework, $_PLUGINS;

		static $tmpl							=	array();

		if ( ! $files ) {
			$files								=	array();
		} elseif ( ! is_array( $files ) ) {
			$files								=	array( $files );
		}

		$id										=	md5( serialize( array( $files, $loadGlobal, $loadHeader ) ) );

		if ( ! isset( $tmpl[$id] ) ) {
			static $plugin						=	null;
			static $params						=	null;

			if ( ! $plugin ) {
				$plugin							=	$_PLUGINS->getLoadedPlugin( 'user', 'cbgallery' );

				if ( ! $plugin ) {
					return;
				}

				$params							=	self::getGlobalParams();
			}

			$livePath							=	$_PLUGINS->getPluginLivePath( $plugin );
			$absPath							=	$_PLUGINS->getPluginPath( $plugin );

			$template							=	$params->getString( 'general_template', 'default' );
			$paths								=	array( 'global_css' => null, 'php' => null, 'css' => null, 'js' => null, 'override_css' => null );

			foreach ( $files as $file ) {
				$file							=	preg_replace( '/[^-a-zA-Z0-9_]/', '', $file );
				$globalCss						=	'/templates/' . $template . '/template.css';
				$overrideCss					=	'/templates/' . $template . '/override.css';

				if ( $file ) {
					$php						=	$absPath . '/templates/' . $template . '/' . $file . '.php';
					$css						=	'/templates/' . $template . '/' . $file . '.css';
					$js							=	'/templates/' . $template . '/' . $file . '.js';
				} else {
					$php						=	null;
					$css						=	null;
					$js							=	null;
				}

				if ( $loadGlobal && $loadHeader ) {
					if ( ! file_exists( $absPath . $globalCss ) ) {
						$globalCss				=	'/templates/default/template.css';
					}

					if ( file_exists( $absPath . $globalCss ) ) {
						$_CB_framework->document->addHeadStyleSheet( $livePath . $globalCss );

						$paths['global_css']	=	$livePath . $globalCss;
					}
				}

				if ( $file ) {
					if ( ! file_exists( $php ) ) {
						$php					=	$absPath . '/templates/default/' . $file . '.php';
					}

					if ( file_exists( $php ) ) {
						require_once( $php );

						$paths['php']			=	$php;
					}

					if ( $loadHeader ) {
						if ( ! file_exists( $absPath . $css ) ) {
							$css				=	'/templates/default/' . $file . '.css';
						}

						if ( file_exists( $absPath . $css ) ) {
							$_CB_framework->document->addHeadStyleSheet( $livePath . $css );

							$paths['css']		=	$livePath . $css;
						}

						if ( ! file_exists( $absPath . $js ) ) {
							$js					=	'/templates/default/' . $file . '.js';
						}

						if ( file_exists( $absPath . $js ) ) {
							$_CB_framework->document->addHeadScriptUrl( $livePath . $js );

							$paths['js']		=	$livePath . $js;
						}
					}
				}

				if ( $loadGlobal && $loadHeader ) {
					if ( file_exists( $absPath . $overrideCss ) ) {
						$_CB_framework->document->addHeadStyleSheet( $livePath . $overrideCss );

						$paths['override_css']	=	$livePath . $overrideCss;
					}
				}
			}

			$tmpl[$id]							=	$paths;
		}
	}

	/**
	 * Returns the current return url or generates one from current page
	 *
	 * @param bool|false $current
	 * @param bool|false $raw
	 * @return null|string
	 */
	public static function getReturn( $current = false, $raw = false )
	{
		static $cache				=	array();

		if ( ! isset( $cache[$current] ) ) {
			$url					=	null;

			if ( $current ) {
				$returnUrl			=	Application::Input()->getBase64( 'get/return' );

				if ( $returnUrl ) {
					$returnUrl		=	base64_decode( $returnUrl );

					if ( \JUri::isInternal( $returnUrl ) || ( $returnUrl[0] === '/' ) ) {
						$url		=	$returnUrl;
					}
				}
			} else {
				$isHttps			=	( isset( $_SERVER['HTTPS'] ) && ( ! empty( $_SERVER['HTTPS'] ) ) && ( $_SERVER['HTTPS'] !== 'off' ) );
				$returnUrl			=	'http' . ( $isHttps ? 's' : '' ) . '://' . $_SERVER['HTTP_HOST'];

				if ( ( ! empty( $_SERVER['PHP_SELF'] ) ) && ( ! empty( $_SERVER['REQUEST_URI'] ) ) ) {
					$returnUrl		.=	$_SERVER['REQUEST_URI'];
				} else {
					$returnUrl		.=	$_SERVER['SCRIPT_NAME'];

					if ( isset( $_SERVER['QUERY_STRING'] ) && ( ! empty( $_SERVER['QUERY_STRING'] ) ) ) {
						$returnUrl	.=	'?' . $_SERVER['QUERY_STRING'];
					}
				}

				$url				=	cbUnHtmlspecialchars( preg_replace( '/[\\\"\\\'][\\s]*javascript:(.*)[\\\"\\\']/', '""', preg_replace( '/eval\((.*)\)/', '', htmlspecialchars( urldecode( $returnUrl ) ) ) ) );
			}

			$cache[$current]		=	$url;
		}

		$return						=	$cache[$current];

		if ( ( ! $raw ) && $return ) {
			$return					=	base64_encode( $return );
		}

		return $return;
	}

	/**
	 * Redirects to the return url if available otherwise to the url specified
	 *
	 * @param string      $url
	 * @param null|string $message
	 * @param string      $messageType
	 */
	public static function returnRedirect( $url, $message = null, $messageType = 'message' )
	{
		$returnUrl		=	self::getReturn( true, true );

		cbRedirect( ( $returnUrl ? $returnUrl : $url ), $message, $messageType );
	}

	/**
	 * Returns file size formatted from bytes
	 *
	 * @param int $bytes
	 * @param int $decimals
	 * @return string
	 */
	public static function getFormattedFileSize( $bytes, $decimals = 0 )
	{
		if ( $bytes >= 1099511627776 ) {
			return CBTxt::T( 'FILESIZE_FORMATTED_TB', '%%COUNT%% TB|%%COUNT%% TBs', array( '%%COUNT%%' => (float) number_format( $bytes / 1099511627776, $decimals, '.', '' ) ) );
		}

		if ( $bytes >= 1073741824 ) {
			return CBTxt::T( 'FILESIZE_FORMATTED_GB', '%%COUNT%% GB|%%COUNT%% GBs', array( '%%COUNT%%' => (float) number_format( $bytes / 1073741824, $decimals, '.', '' ) ) );
		}

		if ( $bytes >= 1048576 ) {
			return CBTxt::T( 'FILESIZE_FORMATTED_MB', '%%COUNT%% MB|%%COUNT%% MBs', array( '%%COUNT%%' => (float) number_format( $bytes / 1048576, $decimals, '.', '' ) ) );
		}

		if ( $bytes >= 1024 ) {
			return CBTxt::T( 'FILESIZE_FORMATTED_KB', '%%COUNT%% KB|%%COUNT%% KBs', array( '%%COUNT%%' => (float) number_format( $bytes / 1024, $decimals, '.', '' ) ) );
		}

		return CBTxt::T( 'FILESIZE_FORMATTED_B', '%%COUNT%% B|%%COUNT%% Bs', array( '%%COUNT%%' => (float) number_format( $bytes, $decimals, '.', '' ) ) );
	}

	/**
	 * Returns the type based off extension
	 *
	 * @param string|ItemTable $extension
	 * @param null|Gallery     $gallery
	 * @param null|string      $method
	 * @param bool             $access
	 * @return string
	 */
	public static function getExtensionType( $extension, $gallery = null, $method = null, $access = true )
	{
		if ( $extension instanceof ItemTable ) {
			$extension		=	$extension->extension();
		}

		// If a gallery is being checked allow file type overloading
		if ( $gallery && in_array( $extension, self::getExtensions( 'files', $gallery, $method, $access ), true ) ) {
			return 'files';
		}

		if ( in_array( $extension, self::getExtensions( 'photos', $gallery, $method, $access ), true ) ) {
			return 'photos';
		}

		if ( in_array( $extension, self::getExtensions( 'videos', $gallery, $method, $access ), true ) ) {
			return 'videos';
		}

		if ( in_array( $extension, self::getExtensions( 'music', $gallery, $method, $access ), true ) ) {
			return 'music';
		}

		// No gallery being displayed so just check the true type instead of overloaded type
		if ( ( ! $gallery ) && in_array( $extension, self::getExtensions( 'files', $gallery, $method, $access ), true ) ) {
			return 'files';
		}

		return null;
	}

	/**
	 * Returns a list of extensions supported by the provided gallery type
	 *
	 * @param string|ItemTable $type
	 * @param null|Gallery     $gallery
	 * @param null|string      $method
	 * @param bool             $access
	 * @return array
	 */
	public static function getExtensions( $type, $gallery = null, $method = null, $access = true )
	{
		$params						=	self::getGlobalParams();

		if ( $gallery ) {
			$params					=	$gallery;
		}

		if ( $type instanceof ItemTable ) {
			$type					=	$type->getString( 'type' );
		}

		$photos						=	array( 'jpg', 'jpeg', 'gif', 'png' );

		if ( ( PHP_VERSION_ID >= 70300 ) && ( $method === 'upload' ) ) {
			$photos[]				=	'webp';
		}

		if ( $method === 'client' ) {
			$method					=	'upload';

			if ( $params->getBool( 'photos_client_resize', true ) ) {
				$photos[]			=	'webp';
				$photos[]			=	'avif';
			}
		}

		$extensions					=	$params->getString( 'files_extensions', 'zip,rar,doc,pdf,txt,xls' );

		$files						=	explode( ',', $extensions );

		$videos						=	array( 'mp4', 'ogv', 'ogg', 'webm', 'm4v' );

		if ( $method === 'link' ) {
			$videos[]				=	'youtube';
			$videos[]				=	'vimeo';
			$videos[]				=	'facebook';
		} elseif ( self::getGlobalParams()->getBool( 'videos_mov', false ) ) {
			$videos[]				=	'mov';
		}

		$music						=	array( 'mp3', 'oga', 'weba', 'wav', 'm4a' );

		if ( $gallery && $access ) {
			if ( ! self::canCreateItems( 'photos', $method, $gallery ) ) {
				$photos				=	array();
			}

			if ( ! self::canCreateItems( 'files', $method, $gallery ) ) {
				$files				=	array();
			}

			if ( ! self::canCreateItems( 'videos', $method, $gallery ) ) {
				$videos				=	array();
			}

			if ( ! self::canCreateItems( 'music', $method, $gallery ) ) {
				$music				=	array();
			}
		}

		switch( $type ) {
			case 'photos':
				return $photos;
			case 'files':
				return $files;
			case 'videos':
				return $videos;
			case 'music':
				return $music;
			case 'all':
				return array_unique( array_merge( $photos, $files, $videos, $music ) );
		}

		return array();
	}

	/**
	 * Returns a list of mimetypes based off extension
	 *
	 * @param array|string|ItemTable $extensions
	 * @return array|string
	 */
	public static function getMimeTypes( $extensions )
	{
		if ( $extensions instanceof ItemTable ) {
			$extensions			=	$extensions->extension();
		}

		/** @var array $mimeTypes */
		$mimeTypes				=	cbGetMimeFromExt( $extensions );

		if ( is_array( $extensions ) ) {
			if ( in_array( 'm4v', $extensions, true ) ) {
				$mimeTypes[]	=	'video/mp4';
			}

			if ( in_array( 'youtube', $extensions, true ) ) {
				$mimeTypes[]	=	'video/youtube';
				$mimeTypes[]	=	'video/x-youtube';
			}

			if ( in_array( 'vimeo', $extensions, true ) ) {
				$mimeTypes[]	=	'video/vimeo';
				$mimeTypes[]	=	'video/x-vimeo';
			}

			if ( in_array( 'facebook', $extensions, true ) ) {
				$mimeTypes[]	=	'video/facebook';
				$mimeTypes[]	=	'video/x-facebook';
			}

			if ( in_array( 'mp3', $extensions, true ) ) {
				$mimeTypes[]	=	'audio/mp3';
			}

			if ( in_array( 'm4a', $extensions, true ) ) {
				$mimeTypes[]	=	'audio/mp4';
			}

			if ( in_array( 'avif', $extensions, true ) ) {
				$mimeTypes[]	=	'image/avif	';
			}
		} else {
			/** @var string $mimeTypes */
			if ( $extensions === 'm4v' ) {
				$mimeTypes		=	'video/mp4';
			} elseif ( $extensions === 'youtube' ) {
				$mimeTypes		=	'video/x-youtube';
			} elseif ( $extensions === 'vimeo' ) {
				$mimeTypes		=	'video/x-vimeo';
			} elseif ( $extensions === 'facebook' ) {
				$mimeTypes		=	'video/x-facebook';
			} elseif ( $extensions === 'mp3' ) {
				$mimeTypes		=	'audio/mp3';
			} elseif ( $extensions === 'm4a' ) {
				$mimeTypes		=	'audio/mp4';
			} elseif ( $extensions === 'avif' ) {
				$mimeTypes		=	'image/avif	';
			}
		}

		if ( is_array( $mimeTypes ) ) {
			$mimeTypes			=	array_unique( $mimeTypes );
		}

		return $mimeTypes;
	}

	/**
	 * Try to find the extension from an upload object
	 *
	 * @param ParamsInterface $upload
	 * @return null|string
	 */
	public static function getUploadExtension( $upload )
	{
		$name				=	$upload->getString( 'name' );
		$extension			=	null;

		if ( $name ) {
			$extension		=	strtolower( preg_replace( '/[^-a-zA-Z0-9_]/', '', pathinfo( parse_url( $name, PHP_URL_PATH ), PATHINFO_EXTENSION ) ) );
		}

		if ( ! $extension ) {
			$extension		=	self::getExtensionFromMimeType( $upload->getString( 'type' ) );
		}

		return $extension;
	}

	/**
	 * @param string $mimeType
	 * @return null|string
	 */
	public static function getExtensionFromMimeType( $mimeType )
	{
		$extension							=	null;

		if ( $mimeType === 'video/mp4' ) {
			$extension						=	'm4v';
		} elseif ( in_array( $mimeType, array( 'video/youtube', 'video/x-youtube' ), true ) ) {
			$extension						=	'youtube';
		} elseif ( in_array( $mimeType, array( 'video/vimeo', 'video/x-vimeo' ), true ) ) {
			$extension						=	'vimeo';
		} elseif ( in_array( $mimeType, array( 'video/facebook', 'video/x-facebook' ), true ) ) {
			$extension						=	'facebook';
		} elseif ( $mimeType === 'audio/mp3' ) {
			$extension						=	'mp3';
		} elseif ( $mimeType === 'audio/mp4' ) {
			$extension						=	'm4a';
		} elseif ( $mimeType === 'image/avif' ) {
			$extension						=	'm4a';
		} else {
			foreach ( cbGetMimeMap() as $ext => $type ) {
				if ( is_array( $type ) ) {
					foreach ( $type as $subExt => $subType ) {
						if ( $mimeType === $subType ) {
							$extension		=	$subExt;

							break 2;
						}
					}
				} elseif ( $mimeType === $type ) {
					$extension				=	$ext;

					break;
				}
			}
		}

		return $extension;
	}

	/**
	 * Creates a directory path from base to user and type
	 *
	 * @param string      $basePath
	 * @param int|null    $userId
	 * @param string|null $type
	 */
	public static function createDirectory( $basePath, $userId = null, $type = null )
	{
		global $_CB_framework;

		if ( ! $basePath ) {
			return;
		}

		$indexPath					=	$_CB_framework->getCfg( 'absolute_path' ) . '/components/com_comprofiler/plugin/user/plug_cbgallery/index.html';

		if ( ! is_dir( $basePath ) ) {
			$oldMask				=	@umask( 0 );

			if ( @mkdir( $basePath, 0755, true ) ) {
				@umask( $oldMask );
				@chmod( $basePath, 0755 );

				if ( ! file_exists( $basePath . '/index.html' ) ) {
					@copy( $indexPath, $basePath . '/index.html' );
					@chmod( $basePath . '/index.html', 0755 );
				}
			} else {
				@umask( $oldMask );
			}
		}

		if ( ! file_exists( $basePath . '/.htaccess' ) ) {
			file_put_contents( $basePath . '/.htaccess', 'deny from all' );
		}

		if ( $userId !== null ) {
			$userPath				=	$basePath . '/' . (int) $userId;

			if ( ! is_dir( $userPath ) ) {
				$oldMask			=	@umask( 0 );

				if ( @mkdir( $userPath, 0755, true ) ) {
					@umask( $oldMask );
					@chmod( $userPath, 0755 );

					if ( ! file_exists( $userPath . '/index.html' ) ) {
						@copy( $indexPath, $userPath . '/index.html' );
						@chmod( $userPath . '/index.html', 0755 );
					}
				} else {
					@umask( $oldMask );
				}
			}

			if ( $type !== null ) {
				$typePath			=	$userPath . '/' . $type;

				if ( ! is_dir( $typePath ) ) {
					$oldMask		=	@umask( 0 );

					if ( @mkdir( $typePath, 0755, true ) ) {
						@umask( $oldMask );
						@chmod( $typePath, 0755 );

						if ( ! file_exists( $typePath . '/index.html' ) ) {
							@copy( $indexPath, $typePath . '/index.html' );
							@chmod( $typePath . '/index.html', 0755 );
						}
					} else {
						@umask( $oldMask );
					}
				}
			}
		}
	}

	/**
	 * Reloads page headers for ajax responses
	 *
	 * @return null|string
	 */
	public static function reloadHeaders()
	{
		global $_CB_framework;

		if ( Application::Input()->getString( 'format' ) !== 'raw' ) {
			return null;
		}

		$_CB_framework->getAllJsPageCodes();

		// Reset meta headers as they can't be used inline anyway:
		$_CB_framework->document->_head['metaTags']		=	array();

		// Reset custom headers as we can't guarantee they can output inline:
		$_CB_framework->document->_head['custom']		=	array();

		// Remove all non-jQuery scripts as they'll likely just cause errors due to redeclaration:
		foreach( $_CB_framework->document->_head['scriptsUrl'] as $url => $script ) {
			if ( ( strpos( $url, 'jquery.' ) === false ) || ( strpos( $url, 'migrate' ) !== false ) ) {
				unset( $_CB_framework->document->_head['scriptsUrl'][$url] );
			}
		}

		$header				=	$_CB_framework->document->outputToHead();

		if ( ! $header ) {
			return null;
		}

		$return				=	'<div class="galleryHeaders" style="position: absolute; display: none; height: 0; width: 0; z-index: -999;">';

		if ( ( strpos( $header, 'cbjQuery' ) !== false ) && ( strpos( $header, 'window.$ = cbjQuery;' ) === false ) ) {
			// The jQuery pointer is missing so lets add it so jQuery rebinds correctly:
			$return			.=		'<script type="text/javascript">window.$ = cbjQuery; window.jQuery = cbjQuery;</script>';
		}

		$return				.=		$header
							.	'</div>';

		return $return;
	}

	/**
	 * Returns an items type translated
	 *
	 * @param string|ItemTable $type
	 * @param bool             $plural
	 * @param bool             $lowercase
	 * @return string
	 */
	public static function translateType( $type, $plural = false, $lowercase = false )
	{
		if ( $type instanceof ItemTable ) {
			$type			=	$type->getString( 'type' );
		}

		switch( $type ) {
			case 'photos':
				$string		=	( $plural ? CBTxt::T( 'Photos' ) : CBTxt::T( 'Photo' ) );
				break;
			case 'videos':
				$string		=	( $plural ? CBTxt::T( 'Videos' ) : CBTxt::T( 'Video' ) );
				break;
			case 'music':
				$string		=	CBTxt::T( 'Music' );
				break;
			default:
				$string		=	( $plural ? CBTxt::T( 'Files' ) : CBTxt::T( 'File' ) );
				break;
		}

		if ( $lowercase ) {
			$string			=	cbutf8_strtolower( $string );
		}

		return $string;
	}

	/**
	 * Returns the fontawesome icon based off type
	 *
	 * @param string|ItemTable $type
	 * @return string
	 */
	public static function getTypeIcon( $type )
	{
		$extension		=	null;

		if ( $type instanceof ItemTable ) {
			$extension	=	$type->extension();
			$type		=	$type->getString( 'type' );
		}

		switch ( $type ) {
			case 'photos':
				return 'fa-picture-o';
			case 'videos':
				return 'fa-play';
			case 'music':
				return 'fa-volume-up';
		}

		if ( $extension ) {
			return self::getExtensionIcon( $extension );
		}

		return 'fa-file-o';
	}

	/**
	 * Returns the fontawesome icon based off extension and that extensions mimetype
	 *
	 * @param string|ItemTable $extension
	 * @return string
	 */
	public static function getExtensionIcon( $extension )
	{
		if ( $extension instanceof ItemTable ) {
			$extension					=	$extension->extension();
		}

		$type							=	'fa-file-o';

		if ( ! $extension ) {
			return $type;
		}

		if ( $extension === 'youtube' ) {
			return 'fa-youtube';
		}

		if ( $extension === 'vimeo' ) {
			return 'fa-vimeo';
		}

		if ( $extension === 'facebook' ) {
			return 'fa-facebook';
		}

		static $cache					=	array();

		if ( ! isset( $cache[$extension] ) ) {
			$mimeParts					=	explode( '/', CBGallery::getMimeTypes( $extension ) );

			switch ( $mimeParts[0] ) {
				case 'video':
					$type				=	'fa-file-video-o';
					break;
				case 'audio':
					$type				=	'fa-file-audio-o';
					break;
				case 'image':
					$type				=	'fa-file-image-o';
					break;
				default:
					switch ( $extension ) {
						case 'txt':
							$type		=	'fa-file-text-o';
							break;
						case 'pdf':
							$type		=	'fa-file-pdf-o';
							break;
						case 'zip':
						case '7z':
						case 'rar':
						case 'tar':
						case 'iso':
							$type		=	'fa-file-archive-o';
							break;
						case 'asp':
						case 'js':
						case 'php':
						case 'xml':
						case 'css':
						case 'java':
						case 'html':
						case 'htm':
						case 'c':
						case 'cs':
						case 'class':
						case 'cpp':
						case 'jar':
						case 'sh':
						case 'json':
						case 'bat':
						case 'cmd':
							$type		=	'fa-file-code-o';
							break;
						case 'ods':
						case 'csv':
						case 'xls':
						case 'xlt':
						case 'xlm':
						case 'xlsx':
						case 'xlsm':
						case 'xltx':
						case 'xltm':
						case 'xlsb':
						case 'xla':
						case 'xlam':
						case 'xll':
						case 'xlw':
							$type		=	'fa-file-excel-o';
							break;
						case 'odt':
						case 'rtf':
						case 'doc':
						case 'dot':
						case 'wbk':
						case 'docx':
						case 'docm':
						case 'dotx':
						case 'dotm':
						case 'docb':
							$type		=	'fa-file-word-o';
							break;
						case 'odp':
						case 'ppt':
						case 'pot':
						case 'pps':
						case 'pptx':
						case 'pptm':
						case 'potx':
						case 'potm':
						case 'ppam':
						case 'ppsx':
						case 'ppsm':
						case 'sldx':
						case 'sldm':
							$type		=	'fa-file-powerpoint-o';
							break;
					}
					break;
			}

			$cache[$extension]			=	$type;
		}

		return $cache[$extension];
	}

	/**
	 * Parses a url for media
	 *
	 * @param string $url
	 * @return array
	 */
	public static function parseUrl( $url )
	{
		global $_CB_framework;

		static $cache											=	[];

		if ( isset( $cache[$url] ) ) {
			return $cache[$url];
		}

		$extension												=	strtolower( pathinfo( parse_url( $url, PHP_URL_PATH ), PATHINFO_EXTENSION ) );

		$media													=	[	'title'			=>	null,
																		'description'	=>	null,
																		'name'			=>	pathinfo( parse_url( $url, PHP_URL_PATH ), PATHINFO_FILENAME ),
																		'mimetype'		=>	self::getMimeTypes( $extension ),
																		'extension'		=>	$extension,
																		'url'			=>	$url,
																		'thumbnail'		=>	null,
																		'date'			=>	null,
																		'modified'		=>	null,
																		'size'			=>	0,
																		'exists'		=>	false
																	];

		$cachePath												=	$_CB_framework->getCfg( 'absolute_path' ) . '/cache/gallery_links';
		$cacheFile												=	$cachePath . '/' . md5( $url ) . '.json';
		$extraCache												=	null;

		if ( file_exists( $cacheFile ) && ( ( ( Application::Date( 'now', 'UTC' )->getTimestamp() - filemtime( $cacheFile ) ) / 3600 ) < 24 ) ) {
			$cachedMedia										=	json_decode( file_get_contents( $cacheFile ), true );

			if ( ( ! $cachedMedia ) || ( ! \is_array( $cachedMedia ) ) ) {
				// Something wrong with the cache file so fallback to default minimum output:
				$cachedMedia									=	$media;
			} elseif ( \count( $cachedMedia ) < \count( $media ) ) {
				// The cached file is missing newer cached items so lets fill them in with defaults:
				foreach ( $media as $k => $v ) {
					if ( isset( $cachedMedia[$k] ) ) {
						continue;
					}

					$cachedMedia[$k]							=	$v;
				}
			}

			$cache[$url]										=	$cachedMedia;

			return $cache[$url];
		}

		$domain													=	preg_replace( '/^(?:(?:\w+\.)*)?(\w+)\..+$/', '\1', parse_url( $url, PHP_URL_HOST ) );

		if ( ! $domain ) {
			$cache[$url]										=	$media;

			return $cache[$url];
		}

		$paths													=	[	'title'			=>	[	'//meta[@name="og:title"]/@content',
																								'//meta[@name="twitter:title"]/@content',
																								'//meta[@name="title"]/@content',
																								'//meta[@property="og:title"]/@content',
																								'//meta[@property="twitter:title"]/@content',
																								'//meta[@property="title"]/@content',
																								'//oembed/title',
																								'//title'
																							],
																		'description'	=>	[	'//meta[@name="og:description"]/@content',
																								'//meta[@name="twitter:description"]/@content',
																								'//meta[@name="description"]/@content',
																								'//meta[@property="og:description"]/@content',
																								'//meta[@property="twitter:description"]/@content',
																								'//meta[@property="description"]/@content',
																								'//oembed/description'
																							],
																		'thumbnail'		=>	[	'//meta[@name="og:image"]/@content',
																								'//meta[@name="og:image:url"]/@content',
																								'//meta[@name="twitter:image"]/@content',
																								'//meta[@name="image"]/@content',
																								'//meta[@property="og:image"]/@content',
																								'//meta[@property="og:image:url"]/@content',
																								'//meta[@property="twitter:image"]/@content',
																								'//meta[@property="image"]/@content',
																								'//oembed/thumbnail_url'
																							]
																	];

		if ( self::checkDOMDocument() ) {
			try {
				$request											=	new Client();
				$requestUrl											=	$url;

				if ( $domain === 'vimeo' ) {
					$requestUrl										=	'https://vimeo.com/api/oembed.xml?url=' . urlencode( $requestUrl );
				}

				$result												=	$request->get( $requestUrl, [	'timeout'			=>	10,
																										'allow_redirects'	=>	[	'max'				=>	5,
																																	'strict'			=>	false,
																																	'referer'			=>	true,
																																	'protocols'			=>	[ 'http', 'https' ],
																																	'track_redirects'	=>	false
																																]
																										] );

				if ( $result && ( (int) $result->getStatusCode() === 200 ) ) {
					if ( cbGuzzleVersion() >= 6 ) {
						$contentType								=	$result->getHeaderLine( 'Content-Type' );
						$contentDisposition							=	$result->getHeaderLine( 'Content-Disposition' );

						$media['date']								=	$result->getHeaderLine( 'Date' );
						$media['modified']							=	$result->getHeaderLine( 'Last-Modified' );
						$media['size']								=	(int) $result->getHeaderLine( 'Content-Length' );
					} else {
						$contentType								=	(string) $result->getHeader( 'Content-Type' );
						$contentDisposition							=	(string) $result->getHeader( 'Content-Disposition' );

						$media['date']								=	(string) $result->getHeader( 'Date' );
						$media['modified']							=	(string) $result->getHeader( 'Last-Modified' );
						$media['size']								=	(int) $result->getHeader( 'Content-Length' );
					}

					$media['exists']								=	true;

					// We can't find the mimetype or extension from the URL so lets check the headers
					if ( ( $media['mimetype'] === 'application/octet-stream' ) || ( ! $media['mimetype'] ) ) {
						list( $media['mimetype'] )					=	explode( ';', $contentType, 2 );

						foreach ( cbGetMimeMap() as $mimeExt => $mimeType ) {
							if ( \is_array( $mimeType ) ) {
								foreach ( $mimeType as $subExt => $subType ) {
									if ( $media['mimetype'] === $subType ) {
										$media['extension']			=	$subExt;

										break 2;
									}
								}
							} elseif ( $media['mimetype'] === $mimeType ) {
								$media['extension']					=	$mimeExt;

								break;
							}
						}

						if ( preg_match( '/\s*filename\s?=\s?(.*)/', $contentDisposition, $filenameMatches ) ) {
							$filenameParts							=	explode( ';', $filenameMatches[1] );
							$media['name']							=	trim( $filenameParts[0], '"' );
						}
					}

					$extension										=	$media['extension'];

					if ( \in_array( $domain, [ 'youtube', 'youtu', 'vimeo', 'facebook', 'fb' ], true ) ) {
						$body										=	(string) $result->getBody();

						if ( \function_exists( 'mb_convert_encoding' ) ) {
							$body									=	mb_convert_encoding( $body, 'HTML-ENTITIES', 'UTF-8' );
						} else {
							$body									=	'<?xml encoding="UTF-8">' . $body;
						}

						$document									=	@new \DOMDocument();

						@$document->loadHTML( $body );

						$xpath										=	@new \DOMXPath( $document );

						foreach ( $paths as $lookupPath => $nodePaths ) {
							foreach ( $nodePaths as $nodePath ) {
								$nodes								=	@$xpath->query( $nodePath );

								if ( ( $nodes !== false ) && isset( $nodes->length ) && $nodes->length ) {
									foreach ( $nodes as $node ) {
										if ( ! isset( $node->nodeValue ) ) {
											continue;
										}

										$nodeValue					=	Get::clean( $node->nodeValue, GetterInterface::STRING );

										if ( ( ! $nodeValue ) || ( $nodeValue[0] === '/' ) ) {
											continue;
										}

										$media[$lookupPath]			=	$nodeValue;
										break 2;
									}
								}
							}
						}

						if ( \in_array( $domain, [ 'youtube', 'youtu' ], true ) ) {
							$extension								=	'youtube';

							$media['name']							=	null;
							$media['extension']						=	$extension;
						} elseif ( $domain === 'vimeo' ) {
							$extension								=	'vimeo';

							$media['name']							=	null;
							$media['extension']						=	$extension;

							// If we've a custom vimeo video link lets convert it to video id:
							$nodes									=	@$xpath->query( '//oembed/video_id' );
							$vimeoId								=	null;

							if ( ( $nodes !== false ) && isset( $nodes->length ) && $nodes->length ) {
								foreach ( $nodes as $node ) {
									if ( ! isset( $node->nodeValue ) ) {
										continue;
									}

									$vimeoId						=	Get::clean( $node->nodeValue, GetterInterface::INT );
									break;
								}
							}

							if ( $vimeoId ) {
								$media['url']						=	'https://vimeo.com/' . $vimeoId;

								$extraCache							=	$cachePath . '/' . md5( $media['url'] ) . '.json';
							}
						} elseif ( \in_array( $domain, [ 'facebook', 'fb' ], true ) ) {
							if ( ( strpos( $url, '/watch' ) !== false ) || ( strpos( $url, '/videos' ) !== false ) || ( parse_url( $url, PHP_URL_HOST ) === 'fb.watch' ) ) {
								$nodes								=	@$xpath->query( '//meta[@name="twitter:player"]/@content' );
								$videoUrl							=	null;

								if ( ( $nodes !== false ) && isset( $nodes->length ) && $nodes->length ) {
									foreach ( $nodes as $node ) {
										if ( ! isset( $node->nodeValue ) ) {
											continue;
										}

										$videoEmbed					=	[];

										parse_str( parse_url( Get::clean( $node->nodeValue, GetterInterface::STRING ), PHP_URL_QUERY ), $videoEmbed );

										$videoUrl					=	( $videoEmbed['href'] ?? null );
										break;
									}
								}

								if ( $videoUrl ) {
									$extension						=	'facebook';

									$media['name']					=	null;
									$media['extension']				=	$extension;
									$media['url']					=	$videoUrl;

									$extraCache						=	$cachePath . '/' . md5( $media['url'] ) . '.json';
								}
							}
						}

						if ( $extension ) {
							$media['mimetype']						=	self::getMimeTypes( $extension );
						}
					}
				}
			} catch( Exception $e ) {}
		}

		self::createDirectory( $cachePath );

		file_put_contents( $cacheFile, json_encode( $media ) );

		if ( $extraCache ) {
			// If the media URL was changed we want to cache the change so we don't request this data twice
			file_put_contents( $extraCache, json_encode( $media ) );
		}

		$cache[$url]											=	$media;

		return $cache[$url];
	}

	/**
	 * Finds the maximum upload limit allowed by PHP
	 *
	 * @param string $format
	 * @return int
	 */
	public static function getPHPMaxUpload( $format = 'B' )
	{
		if ( ! function_exists( 'ini_get' ) ) {
			return 0;
		}

		static $cache					=	null;

		if ( $cache === null ) {
			$postSize					=	ini_get( 'post_max_size' );
			$postSizeFormat				=	strtoupper( substr( $postSize, -1 ) );

			if ( ! in_array( $postSizeFormat, array( 'P', 'T', 'G', 'M', 'K' ), true ) ) {
				$postSize				=	(int) $postSize;
			} else {
				$postSize				=	(int) substr( $postSize, 0, -1 );

				switch ( $postSizeFormat ) {
					case 'T':
						$postSize		*=	1099511627776;
						break;
					case 'G':
						$postSize		*=	1073741824;
						break;
					case 'M':
						$postSize		*=	1048576;
						break;
					case 'K':
						$postSize		*=	1024;
						break;
				}
			}

			$uploadSize					=	ini_get( 'upload_max_filesize' );
			$uploadSizeFormat			=	strtoupper( substr( $uploadSize, -1 ) );

			if ( ! in_array( $uploadSizeFormat, array( 'P', 'T', 'G', 'M', 'K' ), true ) ) {
				$uploadSize				=	(int) $uploadSize;
			} else {
				$uploadSize				=	(int) substr( $uploadSize, 0, -1 );

				switch ( $uploadSizeFormat ) {
					case 'T':
						$uploadSize		*=	1099511627776;
						break;
					case 'G':
						$uploadSize		*=	1073741824;
						break;
					case 'M':
						$uploadSize		*=	1048576;
						break;
					case 'K':
						$uploadSize		*=	1024;
						break;
				}
			}

			$cache						=	(int) min( $postSize, $uploadSize );
		}

		$uploadLimit					=	$cache;

		if ( $format !== 'B' ) {
			switch ( $format ) {
				case 'TB':
					$uploadLimit		/=	1099511627776;
					break;
				case 'GB':
					$uploadLimit		/=	1073741824;
					break;
				case 'MB':
					$uploadLimit		/=	1048576;
					break;
				case 'KB':
					$uploadLimit		/=	1024;
					break;
			}
		}

		return $uploadLimit;
	}

	/**
	 * @return bool
	 */
	public static function checkDOMDocument()
	{
		if ( ! class_exists( 'DOMDocument' ) ) {
			return false;
		}

		if ( ! class_exists( 'DOMXPath' ) ) {
			return false;
		}

		return true;
	}
}
