<?php
/**
 * Community Builder (TM)
 * @version $Id: $
 * @package CommunityBuilder
 * @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
 * @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
 */

namespace CB\Plugin\Progress\Field;

use CB\Database\Table\FieldTable;
use CB\Database\Table\UserTable;
use CB\Plugin\Progress\Helper;
use CBLib\Application\Application;
use CBLib\Registry\GetterInterface;
use CBLib\Registry\ParamsInterface;
use CBLib\Registry\Registry;
use stdClass;

\defined( 'CBLIB') or die();

class ProgressField extends \cbFieldHandler
{

	/**
	 * Formatter:
	 * Returns a field in specified format
	 *
	 * @param  FieldTable  $field
	 * @param  UserTable   $user
	 * @param  string      $output               'html', 'xml', 'json', 'php', 'csvheader', 'csv', 'rss', 'fieldslist', 'htmledit'
	 * @param  string      $formatting           'tr', 'td', 'div', 'span', 'none',   'table'??
	 * @param  string      $reason               'profile' for user profile view, 'edit' for profile edit, 'register' for registration, 'search' for searches
	 * @param  int         $list_compare_types   IF reason == 'search' : 0 : simple 'is' search, 1 : advanced search with modes, 2 : simple 'any' search
	 * @return mixed
	 */
	public function getFieldRow( &$field, &$user, $output, $formatting, $reason, $list_compare_types )
	{
		if ( ( ! Application::Application()->isClient( 'administrator' ) ) && ( $output === 'html' ) ) {
			if ( $field->params->getBool( 'prg_hide', false ) && ( $this->getProgress( $user, $field ) === 100 ) ) {
				return null;
			}

			if ( $field->params->getBool( 'prg_private', true ) && ( $user->getInt( 'id', 0 ) !== Application::MyUser()->getUserId() ) && ( ! Application::MyUser()->isGlobalModerator() ) ) {
				return null;
			}
		}

		return parent::getFieldRow( $field, $user, $output, $formatting, $reason, $list_compare_types );
	}

	/**
	 * Accessor:
	 * Returns a field in specified format
	 *
	 * @param  FieldTable  $field
	 * @param  UserTable   $user
	 * @param  string      $output               'html', 'xml', 'json', 'php', 'csvheader', 'csv', 'rss', 'fieldslist', 'htmledit'
	 * @param  string      $reason               'profile' for user profile view, 'edit' for profile edit, 'register' for registration, 'search' for searches
	 * @param  int         $list_compare_types   IF reason == 'search' : 0 : simple 'is' search, 1 : advanced search with modes, 2 : simple 'any' search
	 * @return mixed
	 */
	public function getField( &$field, &$user, $output, $reason, $list_compare_types )
	{
		if ( ! $user->getInt( 'id', 0 ) ) {
			return null;
		}

		switch ( $output ) {
			case 'html':
			case 'rss':
			case 'htmledit':
				if ( $reason === 'search' ) {
					return null;
				}

				$complete					=	$this->getProgress( $user, $field );

				if ( ! $complete ) {
					return null;
				}

				$bar						=	$field->params->getBool( 'prg_bar', true );
				$barCompleteness			=	$field->params->getBool( 'prg_bar_completeness', false );
				$barHeight					=	$field->params->getInt( 'prg_bar_height', 0 );
				$barStyle					=	'';

				if ( $field->params->getBool( 'prg_bar_striped', false ) ) {
					$barStyle				.=	' progress-bar-striped';

					if ( $field->params->getBool( 'prg_bar_striped', false ) ) {
						$barStyle			.=	' progress-bar-animated';
					}
				}

				$barColors					=	[];
				$barColor					=	'#007bff';

				if ( $bar ) {
					foreach ( $field->params->subTree( 'prg_bar_colors' ) as $color ) {
						$barColors[$color->getInt( 'complete', 0 )]	=	$color->getString( 'color', '#007bff' );
					}

					krsort( $barColors );

					foreach ( $barColors as $percet => $color ) {
						if ( $complete >= $percet ) {
							$barColor		=	$color;
							break;
						}
					}
				}

				$checklistDisplay			=	$field->params->getInt( 'prg_checklist', 0 );
				$checklistColumns			=	$field->params->getInt( 'prg_checklist_columns', 1 );
				$checklistLinks				=	( ( Application::MyUser()->getUserId() === $user->getInt( 'id', 0 ) ) && $field->params->getBool( 'prg_checklist_links', false ) );
				$checklist					=	[];

				if ( $checklistDisplay ) {
					$userFields				=	$this->getProgressFields( $user, $field );
					$checklistComplete		=	\in_array( $checklistDisplay, [ 1, 3 ], true );
					$checklistIncomplete	=	\in_array( $checklistDisplay, [ 2, 3 ], true );

					foreach( $userFields as $userField ) {
						if ( $checklistComplete && ( ! $userField->complete ) && ( ! $checklistIncomplete ) ) {
							continue;
						}

						if ( $checklistIncomplete && $userField->complete && ( ! $checklistComplete ) ) {
							continue;
						}

						$checklist[]		=	$userField;
					}
				}

				ob_start();
				require Helper::getTemplate( $field, 'progress' );
				$progress					=	ob_get_clean();

				return $this->formatFieldValueLayout( $this->_formatFieldOutput( $field->getString( 'name', '' ), $progress, $output, false ), $reason, $field, $user );
			default:
				return $this->_formatFieldOutput( $field->getString( 'name', '' ), $this->getProgress( $user, $field ), $output, false );
		}
	}

	/**
	 * @param UserTable  $user
	 * @param FieldTable $field
	 * @return stdClass[]
	 */
	private function getProgressFields( UserTable $user, FieldTable $field ): array
	{
		$cbFields				=	$field->params->getString( 'prg_fields', '' );

		if ( ! $cbFields ) {
			return [];
		}

		$cbFields				=	explode( '|*|', $cbFields );

		cbArrayToInts( $cbFields );

		$userFields				=	$this->getUserFields( $user, $cbFields, $field );

		if ( ! $userFields ) {
			return [];
		}

		return $userFields;
	}

	/**
	 * @param UserTable  $user
	 * @param FieldTable $field
	 * @return int
	 */
	private function getProgress( UserTable $user, FieldTable $field ): int
	{
		$userFields			=	$this->getProgressFields( $user, $field );

		if ( ! $userFields ) {
			return 0;
		}

		$complete			=	0;
		$worth				=	( 100 / \count( $userFields ) );

		foreach ( $userFields as $userField ) {
			if ( $userField->complete ) {
				$complete	+=	$worth;
			}
		}

		if ( $complete ) {
			if ( $complete > 100 ) {
				$complete	=	100;
			} else {
				$complete	=	round( $complete, 0 );
			}
		}

		return $complete;
	}

	/**
	 * @param  UserTable  $user
	 * @param  int[]      $cbFields
	 * @param  FieldTable $cbField
	 * @return array
	 */
	private function getUserFields( UserTable $user, array $cbFields, FieldTable $cbField ): array
	{
		global $_CB_framework, $_PLUGINS, $_CB_fieldIconDisplayed;

		static $cache								=	[];
		/** @var FieldTable[] $fields */
		static $fields								=	[];

		$userId										=	$user->getInt( 'id', 0 );
		$progress									=	[];

		foreach ( $cbFields as $cbFieldId ) {
			if ( $cbFieldId === $cbField->getInt( 'fieldid', 0 ) ) {
				continue;
			}

			if ( ! isset( $cache[$cbFieldId][$userId] ) ) {
				if ( ! isset( $fields[$cbFieldId] ) ) {
					$loadField						=	new FieldTable();

					$loadField->load( $cbFieldId );

					if ( ! ( $loadField->params instanceof ParamsInterface ) ) {
						$loadField->params			=	new Registry( $loadField->params );
					}

					$fields[$cbFieldId]				=	$loadField;
				}

				$field								=	$fields[$cbFieldId];

				$iconsRendered						=	isset( $_CB_fieldIconDisplayed[$cbFieldId] );

				if ( ( $field->tablecolumns !== '' ) && ( ! trim( $_PLUGINS->callField( $field->getString( 'type', '' ), 'getFieldRow', [ &$field, &$user, 'htmledit', 'none', 'edit', 0 ], $field ) ) ) ) {
					if ( ( ! $iconsRendered ) && isset( $_CB_fieldIconDisplayed[$cbFieldId] ) ) {
						unset( $_CB_fieldIconDisplayed[$cbFieldId] );
					}

					continue;
				}

				if ( ( ! $iconsRendered ) && isset( $_CB_fieldIconDisplayed[$cbFieldId] ) ) {
					unset( $_CB_fieldIconDisplayed[$cbFieldId] );
				}

				$fieldValue							=	$user->get( $field->getString( 'name', '' ) );

				if ( \is_array( $fieldValue ) ) {
					$fieldValue						=	implode( '|*|', $fieldValue );
				}

				if ( ( $fieldValue === null ) && ( ! $field->getString( 'tablecolumns', '' ) ) ) {
					$fieldValue						=	$_PLUGINS->callField( $field->getString( 'type', '' ), 'getFieldRow', [ &$field, &$user, 'php', 'none', 'profile', 0 ], $field );

					if ( \is_array( $fieldValue ) ) {
						$fieldValue					=	array_shift( $fieldValue );

						if ( \is_array( $fieldValue ) ) {
							$fieldValue				=	implode( '|*|', $fieldValue );
						}
					}
				}

				$progressField						=	new stdClass();
				$progressField->id					=	$field->getInt( 'fieldid', 0 );
				$progressField->title				=	$_PLUGINS->callField( $field->getString( 'type', '' ), 'getFieldTitle', [ &$field, &$user, 'html', 'profile' ], $field );

				if ( ! $progressField->title ) {
					$progressField->title			=	$field->getString( 'name', '' );
				}

				$progressField->value				=	$fieldValue;

				switch ( $field->getString( 'type', '' ) ) {
					case 'checkbox':
					case 'terms':
						$progressField->complete	=	( (int) $fieldValue === 1 );
						break;
					case 'progress':
						$progressField->complete	=	( (int) $fieldValue === 100 );
						break;
					default:
						$progressField->complete	=	( ! $this->isUserFieldEmpty( $field, $fieldValue ) );
						break;
				}

				$linkPrefix							=	'';

				switch ( $field->getString( 'type', '' ) ) {
					case 'fieldgroup':
					case 'date':
					case 'datetime':
					case 'time':
						$linkPrefix					=	'cblab';
						break;
				}

				$progressField->link				=	$_CB_framework->userProfileEditUrl( $userId, true, $field->getInt( 'tabid', 0 ) ) . '#' . \urlencode( $linkPrefix . $field->getString( 'name', '' ) );

				$cache[$cbFieldId][$userId]			=	$progressField;
			}

			$progress[]								=	$cache[$cbFieldId][$userId];
		}

		return $progress;
	}

	/**
	 * Checks if the field is empty based off fieldtype
	 *
	 * @param FieldTable $field
	 * @param mixed      $value
	 * @return bool
	 */
	private function isUserFieldEmpty( FieldTable $field, $value ): bool
	{
		if ( ( $value === '' ) || ( $value === null ) ) {
			return true;
		}

		switch ( $field->getString( 'type', '' ) ) {
			case 'fieldgroup':
				return ( $value === '[]' );
			case 'date':
				return ( $value === '0000-00-00' );
			case 'datetime':
				return ( $value === '0000-00-00 00:00:00' );
			case 'time':
				return ( $value === '00:00:00' );
			case 'integer':
			case 'points':
			case 'checkbox':
			case 'terms':
			case 'counter':
				return ( (int) $value === 0 );
			case 'rating':
			case 'float':
				return ( (float) $value === 0.0 );
		}

		return false;
	}
}