<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2022 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\Conditional;

use CB\Plugin\FieldGroups\CBFieldGroups;
use CBLib\Application\Application;
use CBLib\Registry\ParamsInterface;
use CB\Database\Table\FieldTable;
use CB\Database\Table\TabTable;
use CBLib\Registry\Registry;
use CB\Database\Table\UserTable;
use CBLib\Language\CBTxt;

defined('CBLIB') or die();

class CBConditional
{

	/**
	 * @return Registry
	 */
	public static function getGlobalParams()
	{
		global $_PLUGINS;

		static $params	=	null;

		if ( ! $params ) {
			$plugin		=	$_PLUGINS->getLoadedPlugin( 'user', 'cbconditional' );
			$params		=	new Registry();

			if ( $plugin ) {
				$params->load( $plugin->getRaw( 'params' ) );
			}
		}

		return $params;
	}

	/**
	 * Utility function for grabbing a cached field object
	 *
	 * @param int|string $fieldIdOrName
	 * @param int        $profileId
	 * @return FieldTable|null
	 */
	public static function getField( $fieldIdOrName, $profileId = null )
	{
		if ( ! $fieldIdOrName ) {
			return null;
		}

		$userId								=	Application::MyUser()->getUserId();

		if ( $profileId === null ) {
			$profileId						=	$userId;
		}

		static $fields						=	array();

		if ( ! isset( $fields[$profileId][$userId] ) ) {
			$profileUser					=	\CBuser::getInstance( $profileId, false );

			$fields[$profileId][$userId]	=	$profileUser->_getCbTabs( false )->_getTabFieldsDb( null, $profileUser->getUserData(), 'adminfulllist', null, true, true );
		}

		if ( is_string( $fieldIdOrName ) ) {
			$field							=	null;

			foreach ( $fields[$profileId][$userId] as $fld ) {
				/** @var FieldTable $fld */
				if ( $fld->getString( 'name' ) !== $fieldIdOrName ) {
					continue;
				}

				$field						=	$fld;
				break;
			}

			if ( ! $field ) {
				return null;
			}
		} else {
			if ( ! isset( $fields[$profileId][$userId][$fieldIdOrName] ) ) {
				return null;
			}

			$field							=	$fields[$profileId][$userId][$fieldIdOrName];
		}

		if ( ! ( $field->getRaw( 'params' ) instanceof ParamsInterface ) ) {
			$field->set( 'params', new Registry( $field->getRaw( 'params' ) ) );
		}

		return $field;
	}

	/**
	 * Utility function for grabbing a cache tab object
	 *
	 * @param int $tabId
	 * @param int $profileId
	 * @return TabTable|null
	 */
	public static function getTab( $tabId, $profileId = null )
	{
		if ( ! $tabId ) {
			return null;
		}

		$userId								=	Application::MyUser()->getUserId();

		if ( $profileId === null ) {
			$profileId						=	$userId;
		}

		static $tabs						=	array();

		if ( ! isset( $tabs[$profileId][$userId] ) ) {
			$profileUser					=	\CBuser::getInstance( $profileId, false );

			$tabs[$profileId][$userId]		=	$profileUser->_getCbTabs( false )->_getTabsDb( $profileUser->getUserData(), 'adminfulllist' );
		}

		if ( ! isset( $tabs[$profileId][$userId][$tabId] ) ) {
			return null;
		}

		$tab								=	$tabs[$profileId][$userId][$tabId];

		if ( ! ( $tab->getRaw( 'params' ) instanceof ParamsInterface ) ) {
			$tab->set( 'params', new Registry( $tab->getRaw( 'params' ) ) );
		}

		return $tab;
	}

	/**
	 * Utility function for grabbing a tabs cached field objects
	 *
	 * @param int $tabId
	 * @param int $profileId
	 * @return FieldTable[]
	 */
	public static function getTabFields( $tabId, $profileId = null )
	{
		if ( ! $tabId ) {
			return null;
		}

		$userId										=	Application::MyUser()->getUserId();

		if ( $profileId === null ) {
			$profileId								=	$userId;
		}

		static $fields								=	array();

		if ( ! isset( $fields[$tabId][$profileId][$userId] ) ) {
			$profileUser							=	\CBuser::getInstance( $profileId, false );

			$fields[$tabId][$profileId][$userId]	=	$profileUser->_getCbTabs( false )->_getTabFieldsDb( $tabId, $profileUser->getUserData(), 'adminfulllist', null, true, true );
		}

		return $fields[$tabId][$profileId][$userId];
	}

	/**
	 * Check is a tab has been conditioned
	 *
	 * @param int|TabTable $tab
	 * @param string       $reason
	 * @param int          $userId
	 * @param bool         $outputJs
	 * @return bool|int                 0|false = not conditioned, 1|true = conditioned, 2 = static conditioned
	 */
	public static function getTabConditional( $tab, $reason, $userId, $outputJs = false )
	{
		global $_CB_framework;

		if ( ( ! $reason ) || ( $reason === 'adminfulllist' ) ) {
			// Unknown reason or we need full admin list of tabs:
			return false;
		}

		static $cache												=	array();
		static $jsCache												=	array();
		static $jsOutput											=	array();

		if ( ! $tab instanceof TabTable ) {
			$tab													=	self::getTab( $tab, $userId );

			if ( ! $tab ) {
				return false;
			}
		}

		$tabId														=	$tab->getInt( 'tabid', 0 );

		if ( ! isset( $cache[$tabId][$userId][$reason] ) ) {
			if ( ! ( $tab->getRaw( 'params' ) instanceof ParamsInterface ) ) {
				$tab->set( 'params', new Registry( $tab->getRaw( 'params' ) ) );
			}

			$cache[$tabId][$userId][$reason]						=	self::parseConditions( $tab, $reason, $userId, $js );
			$jsCache[$tabId][$userId][$reason]						=	$js;
		}

		$conditioned												=	$cache[$tabId][$userId][$reason];

		if ( $conditioned !== 2 ) {
			// Only output the JS if it's needed (not static):
			$js														=	$jsCache[$tabId][$userId][$reason];

			if ( $outputJs && $js && ( ! isset( $jsOutput[$tabId][$userId][$reason] ) ) ) {
				$_CB_framework->outputCbJQuery( $js, 'cbcondition' );

				$jsOutput[$tabId][$userId][$reason]					=	true;
			}

			if ( $conditioned && ( ! $js ) ) {
				// No JS to output so just treat it as static:
				return 2;
			}
		}

		return $conditioned;
	}

	/**
	 * Check is a field has been conditioned
	 *
	 * @param int|FieldTable $field
	 * @param string         $reason
	 * @param int            $userId
	 * @param bool           $outputJs
	 * @param null|bool      $reset
	 * @return bool|int               0|false = not conditioned, 1|true = conditioned, 2 = static conditioned
	 */
	public static function getFieldConditional( $field, $reason, $userId, $outputJs = false )
	{
		global $_CB_framework;

		if ( ( ! $reason ) || ( $reason === 'adminfulllist' ) ) {
			// Unknown reason or we need full admin list of fields:
			return false;
		}

		static $cache												=	array();
		static $jsCache												=	array();
		static $jsOutput											=	array();

		if ( ! $field instanceof FieldTable ) {
			$field													=	self::getField( $field, $userId );

			if ( ! $field ) {
				return false;
			}
		}

		$fieldId													=	$field->getInt( 'fieldid', 0 );
		$fieldName													=	$field->getString( 'name' );

		if ( ! isset( $cache[$fieldName][$userId][$reason] ) ) {
			if ( ! ( $field->getRaw( 'params' ) instanceof ParamsInterface ) ) {
				$field->set( 'params', new Registry( $field->getRaw( 'params' ) ) );
			}

			// Cache the condition by name since we want to allow cbrepeat fields to be checked:
			$cache[$fieldName][$userId][$reason]					=	self::parseConditions( $field, $reason, $userId, $js );
			// But cache the JS by id as we don't want the JS binding to the same field multiple times (clone code in cbrepeat will take care of rebinding):
			$jsCache[$fieldId][$userId][$reason]					=	$js;
		}

		$conditioned												=	$cache[$fieldName][$userId][$reason];

		if ( $conditioned !== 2 ) {
			// Only output the JS if it's needed (not static):
			$js														=	$jsCache[$fieldId][$userId][$reason];

			if ( $outputJs && $js && ( ! isset( $jsOutput[$fieldId][$userId][$reason] ) ) ) {
				$_CB_framework->outputCbJQuery( $js, 'cbcondition' );

				$jsOutput[$fieldId][$userId][$reason]				=	true;
			}

			if ( $conditioned && ( ! $js ) ) {
				// No JS to output so just treat it as static:
				return 2;
			}
		}

		return $conditioned;
	}

	/**
	 * @param FieldTable|TabTable $object
	 * @param null|bool           $reset
	 * @return bool
	 */
	public static function getReset( $object, $reset = null )
	{
		static $cache		=	array();

		if ( $object instanceof TabTable ) {
			$type			=	'tab';
			$id				=	$object->getInt( 'tabid', 0 );
		} else {
			$type			=	'field';
			$id				=	$object->getInt( 'fieldid', 0 );
		}

		if ( $reset === null ) {
			if ( ! isset( $cache[$type][$id] ) ) {
				if ( $type === 'field' ) {
					$tabId	=	$object->getInt( 'tabid', 0 );

					if ( isset( $cache['tab'][$tabId] ) ) {
						return $cache['tab'][$tabId];
					}
				}

				return self::getGlobalParams()->getBool( 'conditions_reset', true );
			}

			return $cache[$type][$id];
		}

		$cache[$type][$id]	=	$reset;

		return $reset;
	}

	/**
	 * @param FieldTable|TabTable $object
	 * @param string              $reason
	 * @param int                 $userId
	 * @param null|bool           $js
	 * @param null|bool           $reset
	 * @return bool|int
	 */
	private static function parseConditions( $object, $reason, $userId, &$js = null )
	{
		global $_CB_framework, $_CB_database;

		$resetValues											=	$object->params->getString( 'cbconditional_reset', '-1' );

		if ( ( $resetValues !== '' ) && ( $resetValues !== null ) && ( $resetValues !== '-1' ) ) {
			$reset												=	( $resetValues === '1' );

			self::getReset( $object, $reset );
		} else {
			$reset												=	self::getReset( $object );
		}

		$debug													=	$object->params->getString( 'cbconditional_debug', '-1' );

		if ( ( $debug !== '' ) && ( $debug !== null ) && ( $debug !== '-1' ) ) {
			$debug												=	( $debug === '1' );
		} else {
			$debug												=	self::getGlobalParams()->getBool( 'conditions_debug', false );
		}

		$display												=	$object->params->getInt( 'cbconditional_conditioned', 0 );

		$conditioned											=	false;
		$static													=	true;

		if ( $display ) {
			$cbUser												=	\CBuser::getInstance( (int) $userId, false );

			$orMatched											=	false;

			$jsTargets											=	array();
			$jsConditions										=	array();

			$conditionsUsed										=	0;

			if ( $reason === 'search' ) {
				$selectorPrefix									=	'.cbUserListSearchFields ';
			} else {
				$selectorPrefix									=	null;
			}

			foreach ( $object->params->subTree( 'cbconditional_conditions' ) as $orIndex => $orCondition ) {
				/** @var ParamsInterface $orCondition */
				$andMatched										=	true;
				$andConditions									=	array();

				foreach ( $orCondition->subTree( 'condition' ) as $andIndex => $andCondition ) {
					/** @var ParamsInterface $andCondition */
					if ( ! in_array( $reason, explode( '|*|', $andCondition->getString( 'locations', 'register|*|edit|*|profile|*|list' ) ), true ) ) {
						continue;
					}

					$conditionFieldId							=	$andCondition->getString( 'field' );
					$conditionFieldName							=	null;

					if ( ! $conditionFieldId ) {
						continue;
					}

					$operator									=	$andCondition->getInt( 'operator', 0 );

					if ( in_array( $operator, array( 8, 9, 12, 13 ), true ) ) {
						$delimiter								=	$andCondition->getString( 'delimiter' );
					} else {
						$delimiter								=	null;
					}

					$value										=	null;

					if ( ! in_array( $conditionFieldId, array( 'customviewaccesslevels', 'customusergroups', 'customlanguages', 'custommoderators', 'customusers', 'customconnections', 'customsubscriptions' ), true ) ) {
						$value									=	$cbUser->replaceUserVars( $andCondition->getRaw( 'value' ), false, true, self::getSubstitutions(), $andCondition->getBool( 'value_translate', false ) );

						if ( in_array( $operator, array( 6, 7 ), true ) ) {
							$value								=	null;
						}
					}

					$conditionField								=	null;
					$jsType										=	'static';

					switch ( $conditionFieldId ) {
						case 'customvalue':
							$fieldValue							=	$cbUser->replaceUserVars( $andCondition->getRaw( 'field_custom' ), false, true, self::getSubstitutions(), $andCondition->getBool( 'field_custom_translate', false ) );
							break;
						case 'customviewaccesslevels':
							$accessLevels						=	cbToArrayOfInt( explode( '|*|', $andCondition->getString( 'field_viewaccesslevels' ) ) );

							if ( $andCondition->getString( 'user_viewaccesslevels', 'displayed' ) === 'viewer' ) {
								$userAccessLevels				=	Application::MyUser()->getAuthorisedViewLevels();
							} else {
								$userAccessLevels				=	Application::User( (int) $userId )->getAuthorisedViewLevels();
							}

							$fieldValue							=	0;

							foreach ( $accessLevels as $accessLevel ) {
								if ( in_array( $accessLevel, $userAccessLevels, true ) ) {
									$fieldValue					=	1;
									break;
								}
							}

							$operator							=	$andCondition->getInt( 'operator_viewaccesslevels', 0 );
							$value								=	1;
							break;
						case 'customusergroups':
							$userGroups							=	cbToArrayOfInt( explode( '|*|', $andCondition->getString( 'field_usergroups' ) ) );

							if ( $andCondition->getString( 'user_usergroups', 'displayed' ) === 'viewer' ) {
								$userUsergroups					=	Application::MyUser()->getAuthorisedGroups();
							} else {
								$userUsergroups					=	Application::User( (int) $userId )->getAuthorisedGroups();
							}

							$fieldValue							=	0;

							foreach ( $userGroups as $userGroup ) {
								if ( in_array( $userGroup, $userUsergroups, true ) ) {
									$fieldValue					=	1;
									break;
								}
							}

							$operator							=	$andCondition->getInt( 'operator_usergroups', 0 );
							$value								=	1;
							break;
						case 'customlanguages':
							if ( $andCondition->getString( 'from_languages', 'user' ) === 'site' ) {
								$fieldValue						=	Application::Cms()->getLanguageTag();
							} elseif ( $andCondition->getString( 'user_languages', 'displayed' ) === 'viewer' ) {
								$fieldValue						=	\CBuser::getMyUserDataInstance()->getUserLanguage();
							} else {
								$fieldValue						=	$cbUser->getUserData()->getUserLanguage();
								$jsType							=	'language';
							}

							$operator							=	$andCondition->getInt( 'operator_languages', 12 );
							$delimiter							=	'|*|';
							$value								=	$andCondition->getString( 'field_languages' );

							$conditionField						=	self::getField( 'params', $userId );
							$conditionFieldId					=	$conditionField->getInt( 'fieldid', 0 );
							$conditionFieldName					=	$conditionField->getString( 'name' );
							break;
						case 'custommoderators':
							if ( $andCondition->getString( 'user_moderators', 'displayed' ) === 'viewer' ) {
								$fieldValue						=	(int) Application::MyUser()->isGlobalModerator();
							} else {
								$fieldValue						=	(int) Application::User( (int) $userId )->isGlobalModerator();
							}

							$operator							=	$andCondition->getInt( 'operator_moderators', 0 );
							$value								=	1;
							break;
						case 'customusers':
							if ( $andCondition->getString( 'user_users', 'displayed' ) === 'viewer' ) {
								$fieldValue						=	Application::MyUser()->getUserId();
							} else {
								$fieldValue						=	(int) $userId;
							}

							$operator							=	$andCondition->getInt( 'operator_users', 12 );
							$delimiter							=	$andCondition->getString( 'delimiter_users' );

							if ( ! $delimiter ) {
								$delimiter						=	',';
							}

							$value								=	$cbUser->replaceUserVars( $andCondition->getString( 'field_users' ), true, false, array(), false );
							break;
						case 'customconnections':
							$viewerId							=	Application::MyUser()->getUserId();

							if ( $viewerId === (int) $userId ) {
								$fieldValue						=	0;
							} elseif ( $andCondition->getString( 'user_connections', 'displayed' ) === 'viewer' ) {
								$connections					=	new \cbConnection( $viewerId );
								$connection						=	$connections->getConnectionDetails( $viewerId, (int) $userId );
								$fieldValue						=	( ( $connection !== false ) && ( (int) $connection->accepted === 1 ) && ( (int) $connection->pending === 0 ) ? 1 : 0 );
							} else {
								$connections					=	new \cbConnection( (int) $userId );
								$connection						=	$connections->getConnectionDetails( (int) $userId, $viewerId );
								$fieldValue						=	( ( $connection !== false ) && ( (int) $connection->accepted === 1 ) && ( (int) $connection->pending === 0 ) ? 1 : 0 );
							}

							$operator							=	$andCondition->getInt( 'operator_connections', 0 );
							$value								=	1;
							break;
						case 'customcode':
							$code								=	$cbUser->replaceUserVars( $andCondition->getRaw( 'field_code' ), false, true, self::getSubstitutions(), $andCondition->getBool( 'field_code_translate', false ) );

							try {
								$codeField						=	null;
								$codeTab						=	null;

								if ( $object instanceof FieldTable ) {
									$codeField					=	$object;
								} elseif ( $object instanceof TabTable ) {
									$codeTab					=	$object;
								}

								$fieldValue						=	self::outputCode( $code, $codeField, $codeTab, $cbUser->getUserData() );
							} catch ( \Exception $e ) {
								if ( $debug ) {
									$_CB_framework->enqueueMessage( CBTxt::T( 'CONDITION_CODE_FAILED', 'Condition code failed. Error: [error]', array( '[error]' => $e->getMessage() ) ), 'error' );
								}
							}
							break;
						case 'customquery':
							$query								=	$cbUser->replaceUserVars( $andCondition->getRaw( 'field_query' ), array( '\CB\Plugin\Conditional\CBConditional', 'escapeSQL' ), true, self::getSubstitutions(), $andCondition->getBool( 'field_query_translate', false ) );

							try {
								if ( stripos( trim( $query ), 'SELECT' ) === 0 ) {
									$_CB_database->setQuery( $query );

									if ( stripos( trim( $query ), 'SELECT COUNT' ) === 0  ) {
										// Count
										$fieldValue				=	$_CB_database->loadResult();
									} elseif ( in_array( $operator, array( 8, 9 ), true ) ) {
										// Multiple Rows
										$fieldValue				=	$_CB_database->loadResultArray();
									} else {
										if ( preg_match( '/LIMIT (?:0,\s*1|1)/i', trim( $query ) ) ) {
											// Single Row
											$fieldValue			=	$_CB_database->loadResult();
										} else {
											// Multiple Rows
											$fieldValue			=	implode( '|*|', $_CB_database->loadResultArray() );
										}
									}
								}
							} catch ( \Exception $e ) {
								if ( $debug ) {
									$_CB_framework->enqueueMessage( CBTxt::T( 'CONDITION_QUERY_FAILED', 'Condition query failed. Error: [error]', array( '[error]' => $e->getMessage() ) ), 'error' );
								}
							}
							break;
						case 'customsubscriptions':
							if ( $andCondition->getString( 'user_subscriptions', 'displayed' ) === 'viewer' ) {
								$subscriptions					=	self::getCBSubsSubscriptionsStatus( \CBuser::getMyUserDataInstance(), $reason );
							} else {
								$subscriptions					=	self::getCBSubsSubscriptionsStatus( $cbUser->getUserData(), $reason );
							}

							$plans								=	$andCondition->getString( 'plans_subscriptions' );

							if ( $plans ) {
								$plans							=	explode( '|*|', $plans );

								if ( in_array( '', $plans, true ) ) {
									// Any Plan was selected so lets just enforce its selection:
									$plans						=	null;
								}
							}

							$status								=	$andCondition->getString( 'status_subscriptions' );

							if ( $status ) {
								$status							=	explode( '|*|', $status );

								if ( in_array( '', $status, true ) ) {
									// Any Status was selected so lets just enforce its selection:
									$status						=	null;
								}
							}

							$operator							=	$andCondition->getInt( 'operator_subscriptions', 0 );

							if ( ( ( $reason === 'register' ) || ( ( $reason === 'edit' ) && Application::Application()->isClient( 'administrator' ) ) )
								 && ( ( ! $status ) || in_array( 'S', $status, true ) )
							) {
								$static							=	false;
								$jsType							=	'subscription';

								if ( $plans ) {
									$delimiter					=	'|*|';
									$operator					=	( $operator === 0 ? 12 : 13 );
									$value						=	implode( '|*|', $plans );
								} else {
									$operator					=	( $operator === 0 ? 7 : 6 );
								}

								$fieldValue						=	implode( '|*|', array_keys( $subscriptions ) );
							} else {
								$fieldValue						=	0;

								if ( $plans ) {
									// Specific Plan with Any Status
									foreach ( $plans as $planId ) {
										if ( ! isset( $subscriptions[$planId] ) ) {
											continue;
										}

										if ( $status ) {
											// Specific Plan with Specific Status
											if ( ! in_array( $subscriptions[$planId], $status, true ) ) {
												continue;
											}
										}

										$fieldValue				=	1;
										break;
									}
								} elseif ( $status ) {
									// Any Plan with Specific Status
									foreach ( $subscriptions as $subscriptionStatus ) {
										if ( ! in_array( $subscriptionStatus, $status, true ) ) {
											continue;
										}

										$fieldValue				=	1;
										break;
									}
								} elseif ( $subscriptions ) {
									// Any Plan with Any Status
									$fieldValue					=	1;
								}

								$value							=	1;
							}
							break;
						default:
							if ( $andCondition->getString( 'user_fields', 'displayed' ) === 'viewer' ) {
								$fieldUserId					=	Application::MyUser()->getUserId();
							} else {
								$fieldUserId					=	(int) $userId;
							}

							$conditionField						=	self::getField( (int) $conditionFieldId, $fieldUserId );

							if ( ! $conditionField ) {
								continue 2;
							}

							$conditionFieldId					=	$conditionField->getInt( 'fieldid', 0 );
							$conditionFieldName					=	$conditionField->getString( 'name' );

							if ( $object->getBool( '_isGrouped', false )
								 && $object->getRaw( '_fieldGroup' )
								 && in_array( $conditionFieldId, $object->getRaw( '_fieldGroupFields', array() ), true )
							) {
								// This field is inside of a field group so lets point to it instead:
								$user							=	$cbUser->getUserData();

								$fieldGroupField				=	CBFieldGroups::getGroupedFields( $object->getRaw( '_fieldGroup' ), $user, $reason, null, $object->getInt( '_fieldGroupIndex', 0 ), true );

								if ( isset( $fieldGroupField[$conditionFieldName] ) ) {
									$conditionField				=	$fieldGroupField[$conditionFieldName];
									$conditionFieldName			=	$conditionField->getString( 'name' );
								}
							}

							$fieldValue							=	self::getFieldValue( $fieldUserId, $conditionField, $reason, $reset );

							if ( ( $reason === 'edit' ) && ( $fieldUserId !== (int) $userId ) ) {
								// We're conditioning based off viewing user and we're not the profile owner so convert to a static condition:
								$jsType							=	'static';
							} else {
								$jsType							=	'field';
							}
							break;
					}

					if ( $conditionField ) {
						// If there's a field available lets check if it's being output to determine if it's a non-static condition with JS:
						if ( $jsType !== 'static' ) {
							if ( $reason === 'register' ) {
								if ( $conditionField->getInt( 'registration', 1 ) > 0 ) {
									$static						=	false;
								}
							} elseif ( $reason === 'edit' ) {
								if ( $conditionField->getInt( 'edit', 1 ) > 0 ) {
									$static						=	false;
								}
							} elseif ( $reason === 'search' ) {
								if ( $conditionField->getBool( 'searchable', false ) ) {
									$static						=	false;
								}
							}
						}

						if ( $object instanceof FieldTable ) {
							if ( ( ! $static ) && ( $conditionField->getInt( 'fieldid', 0 ) === $object->getInt( 'fieldid', 0 ) ) ) {
								$static							=	true;
								$jsType							=	'static';

								if ( $debug ) {
									// If we're a dynamic condition and we're going to trigger an infinite loop then output a debug error so this can be corrected:
									$_CB_framework->enqueueMessage( CBTxt::T( 'FIELD_CONDITION_SELF', 'Infinite loop detected! Field [field] is conditioning self. This condition will be forced to a static condition to avoid JavaScript errors.', array( '[field]' => $conditionField->getString( 'name' ) ) ), 'error' );
								}
							}
						} elseif ( $object instanceof TabTable ) {
							if ( ( ! $static ) && ( $conditionField->getInt( 'tabid', 0 ) === $object->getInt( 'tabid', 0 ) ) ) {
								$static							=	true;
								$jsType							=	'static';

								if ( $debug ) {
									// If we're a dynamic condition and we're going to trigger an infinite loop then output a debug error so this can be corrected:
									$_CB_framework->enqueueMessage( CBTxt::T( 'FIELD_CONDITION_OWN_TAB', "Infinite loop detected! Field [field] is conditioning own tab [tab]. This condition will be forced to a static condition to avoid JavaScript errors.", array( '[field]' => $conditionField->getString( 'name' ), '[tab]' => $object->getInt( 'tabid', 0 ) ) ), 'error' );
								}
							}
						}
					}

					$isMatched									=	self::getConditionMatch( $conditionField, $fieldValue, $operator, $value, $delimiter );

					if ( $andMatched ) {
						$andMatched								=	$isMatched;
					}

					if ( $debug ) {
						if ( $object instanceof FieldTable ) {
							$extras								=	array(	'[field]'		=>	$object->getString( 'name' ) . ' (' . $object->getInt( 'fieldid', 0 ) . ')',
																			'[and]'			=>	( $andIndex + 1 ),
																			'[or]'			=>	( $orIndex + 1 ),
																			'[input]'		=>	( $fieldValue === null ? 'NULL' : ( is_array( $fieldValue ) ? implode( ( $delimiter ? $delimiter : '|*|' ), $fieldValue ) : $fieldValue ) ),
																			'[operator]'	=>	self::getOperator( $operator ),
																			'[value]'		=>	( $value === null ? 'NULL' : ( is_array( $value ) ? implode( ( $delimiter ? $delimiter : '|*|' ), $value ) : $value ) )
																		);

							if ( $isMatched ) {
								$_CB_framework->enqueueMessage( CBTxt::T( 'FIELD_CONDITION_DEBUG_SUCCESS', 'Field [field] condition [and] (AND) of [or] (OR) matched with: "[input]" [operator] "[value]"', $extras ) );
							} else {
								$_CB_framework->enqueueMessage( CBTxt::T( 'FIELD_CONDITION_DEBUG_FAILED', 'Field [field] condition [and] (AND) of [or] (OR) failed to match with: "[input]" [operator] "[value]"', $extras ), 'error' );
							}
						} elseif ( $object instanceof TabTable ) {
							$extras								=	array(	'[tab]'			=>	$object->getInt( 'tabid', 0 ),
																			'[field]'		=>	$conditionFieldName,
																			'[and]'			=>	( $andIndex + 1 ),
																			'[or]'			=>	( $orIndex + 1 ),
																			'[input]'		=>	( $fieldValue === null ? 'NULL' : ( is_array( $fieldValue ) ? implode( ( $delimiter ? $delimiter : '|*|' ), $fieldValue ) : $fieldValue ) ),
																			'[operator]'	=>	self::getOperator( $operator ),
																			'[value]'		=>	( $value === null ? 'NULL' : ( is_array( $value ) ? implode( ( $delimiter ? $delimiter : '|*|' ), $value ) : $value ) )
																		);

							if ( $isMatched ) {
								$_CB_framework->enqueueMessage( CBTxt::T( 'TAB_CONDITION_DEBUG_SUCCESS', 'Tab [tab] condition [and] (AND) of [or] (OR) matched with: "[input]" ([field]) [operator] "[value]"', $extras ) );
							} else {
								$_CB_framework->enqueueMessage( CBTxt::T( 'TAB_CONDITION_DEBUG_FAILED', 'Tab [tab] condition [and] (AND) of [or] (OR) failed to match with: "[input]" ([field]) [operator] "[value]"', $extras ), 'error' );
							}
						}
					}

					$jsElements									=	array();

					switch ( $jsType ) {
						case 'static':
							$andConditions[]					=	"{"
																.		"type: " . json_encode( $jsType, JSON_HEX_TAG ) . ","
																.		"input: " . json_encode( str_replace( array( "\n", "\r" ), array( "\\n", "\\r" ), ( is_array( $fieldValue ) ? implode( ( $delimiter ? $delimiter : '|*|' ), $fieldValue ) : $fieldValue ) ), JSON_HEX_TAG ) . ","
																.		"operator: " . (int) $operator . ","
																.		( $delimiter ? "delimiter: " . json_encode( $delimiter, JSON_HEX_TAG ) . "," : null )
																.		"value: " . json_encode( str_replace( array( "\n", "\r" ), array( "\\n", "\\r" ), ( is_array( $value ) ? implode( ( $delimiter ? $delimiter : '|*|' ), $value ) : $value ) ), JSON_HEX_TAG )
																.	"}";
							break;
						case 'language':
							$jsTargets[]						=	$selectorPrefix . '#params_language';
							$jsElements[]						=	$selectorPrefix . '#params_language';

							$andConditions[]					=	"{"
																.		"type: " . json_encode( $jsType, JSON_HEX_TAG ) . ","
																.		"element: " . json_encode( implode( ',', $jsElements ), JSON_HEX_TAG ) . ","
																.		"input: " . json_encode( str_replace( array( "\n", "\r" ), array( "\\n", "\\r" ), ( is_array( $fieldValue ) ? implode( ( $delimiter ? $delimiter : '|*|' ), $fieldValue ) : $fieldValue ) ), JSON_HEX_TAG ) . ","
																.		"operator: " . (int) $operator . ","
																.		( $delimiter ? "delimiter: " . json_encode( $delimiter, JSON_HEX_TAG ) . "," : null )
																.		"value: " . json_encode( str_replace( array( "\n", "\r" ), array( "\\n", "\\r" ), ( is_array( $value ) ? implode( ( $delimiter ? $delimiter : '|*|' ), $value ) : $value ) ), JSON_HEX_TAG )
																.	"}";
							break;
						case 'subscription':
							$jsTargets[]						=	$selectorPrefix . '#cbregUpgrades';
							$jsElements[]						=	$selectorPrefix . '#cbregUpgrades';

							$andConditions[]					=	"{"
																.		"type: " . json_encode( $jsType, JSON_HEX_TAG ) . ","
																.		"element: " . json_encode( implode( ',', $jsElements ), JSON_HEX_TAG ) . ","
																.		"input: " . json_encode( str_replace( array( "\n", "\r" ), array( "\\n", "\\r" ), ( is_array( $fieldValue ) ? implode( ( $delimiter ? $delimiter : '|*|' ), $fieldValue ) : $fieldValue ) ), JSON_HEX_TAG ) . ","
																.		"operator: " . (int) $operator . ","
																.		( $delimiter ? "delimiter: " . json_encode( $delimiter, JSON_HEX_TAG ) . "," : null )
																.		"value: " . json_encode( str_replace( array( "\n", "\r" ), array( "\\n", "\\r" ), ( is_array( $value ) ? implode( ( $delimiter ? $delimiter : '|*|' ), $value ) : $value ) ), JSON_HEX_TAG )
																.	"}";
							break;
						case 'field':
						default:
							$jsTargets[]						=	$selectorPrefix . '#cbfr_' . $conditionFieldId;
							$jsTargets[]						=	$selectorPrefix . '#cbfrd_' . $conditionFieldId;
							$jsElements[]						=	$selectorPrefix . '#cbfr_' . $conditionFieldId;
							$jsElements[]						=	$selectorPrefix . '#cbfrd_' . $conditionFieldId;

							$andConditions[]					=	"{"
																.		"type: " . json_encode( $jsType, JSON_HEX_TAG ) . ","
																.		"element: " . json_encode( implode( ',', $jsElements ), JSON_HEX_TAG ) . ","
																.		"input: " . json_encode( str_replace( array( "\n", "\r" ), array( "\\n", "\\r" ), ( is_array( $fieldValue ) ? implode( ( $delimiter ? $delimiter : '|*|' ), $fieldValue ) : $fieldValue ) ), JSON_HEX_TAG ) . ","
																.		"operator: " . (int) $operator . ","
																.		( $delimiter ? "delimiter: " . json_encode( $delimiter, JSON_HEX_TAG ) . "," : null )
																.		"value: " . json_encode( str_replace( array( "\n", "\r" ), array( "\\n", "\\r" ), ( is_array( $value ) ? implode( ( $delimiter ? $delimiter : '|*|' ), $value ) : $value ) ), JSON_HEX_TAG )
																.	"}";
							break;
					}

					$conditionsUsed++;
				}

				if ( ! $andConditions ) {
					// The OR case has no conditions so skip it entirely:
					continue;
				}

				$jsConditions[]									=	"[" . implode( ",", $andConditions ) . "]";

				if ( ( ! $orMatched  ) && $andMatched ) {
					$orMatched									=	true;
				}
			}

			if ( $conditionsUsed ) {
				$conditioned									=	$orMatched;

				if ( $jsConditions ) {
					$showHide									=	array();

					if ( $object instanceof FieldTable ) {
						$fieldId								=	$object->getInt( 'fieldid', 0 );

						$showHide[]								=	$selectorPrefix . '#cbfr_' . $fieldId;
						$showHide[]								=	$selectorPrefix . '#cbfrd_' . $fieldId;

						switch ( $object->getString( 'type' ) ) {
							case 'password':
								$showHide[]						=	$selectorPrefix . '#cbfr_' . $fieldId . '__verify';
								$showHide[]						=	$selectorPrefix . '#cbfrd_' . $fieldId . '__verify';
								$showHide[]						=	$selectorPrefix . '#cbfr_' . $fieldId . '__current';
								$showHide[]						=	$selectorPrefix . '#cbfrd_' . $fieldId . '__current';
								break;
							case 'email':
								$showHide[]						=	$selectorPrefix . '#cbfr_' . $fieldId . '__verify';
								$showHide[]						=	$selectorPrefix . '#cbfrd_' . $fieldId . '__verify';
								break;
						}
					} elseif ( $object instanceof TabTable ) {
						$tabId									=	$object->getInt( 'tabid', 0 );

						$showHide[]								=	$selectorPrefix . '#cbtabpane' . $tabId;
						$showHide[]								=	$selectorPrefix . '#cbtf_' . $tabId;
						$showHide[]								=	$selectorPrefix . '#cbtp_' . $tabId;

						foreach ( self::getTabFields( $tabId, $userId ) as $tabField ) {
							/** @var  FieldTable $tabField */
							$fId								=	$tabField->getInt( 'fieldid', 0 );

							$showHide[]							=	$selectorPrefix . '#cbfr_' . $fId;
							$showHide[]							=	$selectorPrefix . '#cbfrd_' . $fId;

							switch ( $tabField->getString( 'type' ) ) {
								case 'password':
									$showHide[]					=	$selectorPrefix . '#cbfr_' . $fId . '__verify';
									$showHide[]					=	$selectorPrefix . '#cbfrd_' . $fId . '__verify';
									$showHide[]					=	$selectorPrefix . '#cbfr_' . $fId . '__current';
									$showHide[]					=	$selectorPrefix . '#cbfrd_' . $fId . '__current';
									break;
								case 'email':
									$showHide[]					=	$selectorPrefix . '#cbfr_' . $fId . '__verify';
									$showHide[]					=	$selectorPrefix . '#cbfrd_' . $fId . '__verify';
									break;
							}
						}
					}

					if ( $showHide ) {
						$js										.=	"$( " . ( $jsTargets ? json_encode( implode( ',', $jsTargets ), JSON_HEX_TAG ) : 'window' ) . " ).cbcondition({"
																.		"conditions: [" . implode( ",", $jsConditions ) . "],"
																.		( $display === 1 ? "show: " . json_encode( $showHide, JSON_HEX_TAG ) . "," : "hide: " . json_encode( $showHide, JSON_HEX_TAG ) . "," )
																.		"reset: " . (int) $reset . ","
																.		"debug: " . (int) $debug
																.	"});";
					}
				}

				if ( $display === 1 ) {
					// Since we're displaying on match we need to reverse the condition:
					if ( $conditioned ) {
						$conditioned							=	false;
					} else {
						$conditioned							=	true;
					}
				}
			}
		}

		if ( $conditioned && $static ) {
			// Set the condition to static since we've no actual need for the JS:
			$conditioned										=	2;
		}

		return $conditioned;
	}

	/**
	 * Compares condition values based off operator
	 *
	 * @param null|FieldTable $field
	 * @param string          $value
	 * @param int             $operator
	 * @param string          $input
	 * @param null|string     $delimiter
	 * @return bool
	 */
	private static function getConditionMatch( $field, $value, $operator, $input, $delimiter = null )
	{
		$valueIsArray		=	false;

		if ( is_array( $value ) ) {
			$valueIsArray	=	true;
			$value			=	implode( ( $delimiter ? $delimiter : '|*|' ), $value );
		}

		$inputIsArray		=	false;

		if ( is_array( $input ) ) {
			$inputIsArray	=	true;
			$input			=	implode( ( $delimiter ? $delimiter : '|*|' ), $input );
		}

		$value				=	trim( $value );
		$input				=	trim( $input );

		switch ( $operator ) {
			case 1:
				/** @noinspection TypeUnsafeComparisonInspection */
				$match		=	( $value != $input );
				break;
			case 2:
				$match		=	( $value > $input );
				break;
			case 3:
				$match		=	( $value < $input );
				break;
			case 4:
				$match		=	( $value >= $input );
				break;
			case 5:
				$match		=	( $value <= $input );
				break;
			case 6:
				if ( $field ) {
					$match	=	self::isFieldEmpty( $field, $value );
				} else {
					$match	=	( ! $value );
				}
				break;
			case 7:
				if ( $field ) {
					$match	=	( ! self::isFieldEmpty( $field, $value ) );
				} else {
					$match	=	( $value );
				}
				break;
			case 8:
				if ( $delimiter ) {
					$values			=	explode( $delimiter, $value );

					if ( $inputIsArray ) {
						foreach ( explode( $delimiter, $input ) as $v ) {
							$match	=	( in_array( $v, $values, true ) );

							if ( $match ) {
								break;
							}
						}
					} else {
						$match		=	( in_array( $input, $values, true ) );
					}
				} else {
					if ( $input === '' ) {
						// Can't have an empty needle so fallback to simple equal to check:
						return self::getConditionMatch( $field, $value, 0, $input, $delimiter );
					}

					$match	=	( stripos( $value, $input ) !== false );
				}
				break;
			case 9:
				if ( $delimiter ) {
					$values			=	explode( $delimiter, $value );

					if ( $inputIsArray ) {
						foreach ( explode( $delimiter, $input ) as $v ) {
							$match	=	( ! in_array( $v, $values, true ) );

							if ( $match ) {
								break;
							}
						}
					} else {
						$match		=	( ! in_array( $input, $values, true ) );
					}
				} else {
					if ( $input === '' ) {
						// Can't have an empty needle so fallback to simple not equal to check:
						return self::getConditionMatch( $field, $value, 1, $input, $delimiter );
					}

					$match	=	( stripos( $value, $input ) === false );
				}
				break;
			case 10:
				if ( $input === '' ) {
					// Can't have an empty regexp so fallback to simple equal to check:
					return self::getConditionMatch( $field, $value, 0, $input, $delimiter );
				}

				$match		=	( preg_match( $input, $value ) );
				break;
			case 11:
				if ( $input === '' ) {
					// Can't have an empty regexp so fallback to simple not equal to check:
					return self::getConditionMatch( $field, $value, 1, $input, $delimiter );
				}

				$match		=	( ! preg_match( $input, $value ) );
				break;
			case 12:
				if ( $delimiter ) {
					$inputs			=	explode( $delimiter, $input );

					if ( $valueIsArray ) {
						foreach ( explode( $delimiter, $value ) as $v ) {
							$match	=	( in_array( $v, $inputs, true ) );

							if ( $match ) {
								break;
							}
						}
					} else {
						$match		=	( in_array( $value, $inputs, true ) );
					}
				} else {
					if ( $value === '' ) {
						// Can't have an empty needle so fallback to simple equal to check:
						return self::getConditionMatch( $field, $value, 0, $input, $delimiter );
					}

					$match	=	( stripos( $input, $value ) !== false );
				}
				break;
			case 13:
				if ( $delimiter ) {
					$inputs			=	explode( $delimiter, $input );

					if ( $valueIsArray ) {
						foreach ( explode( $delimiter, $value ) as $v ) {
							$match	=	( ! in_array( $v, $inputs, true ) );

							if ( $match ) {
								break;
							}
						}
					} else {
						$match		=	( ! in_array( $value, $inputs, true ) );
					}
				} else {
					if ( $value === '' ) {
						// Can't have an empty needle so fallback to simple not equal to check:
						return self::getConditionMatch( $field, $value, 1, $input, $delimiter );
					}

					$match	=	( stripos( $input, $value ) === false );
				}
				break;
			case 0:
			default:
				/** @noinspection TypeUnsafeComparisonInspection */
				$match		=	( $value == $input );
				break;
		}

		return (bool) $match;
	}

	/**
	 * Returns human readable text for operator
	 *
	 * @param int $operator
	 * @return string
	 */
	private static function getOperator( $operator )
	{
		switch ( $operator ) {
			case 1:
				$operator	=	CBTxt::T( 'Not Equal To' );
				break;
			case 2:
				$operator	=	CBTxt::T( 'Greater Than' );
				break;
			case 3:
				$operator	=	CBTxt::T( 'Less Than' );
				break;
			case 4:
				$operator	=	CBTxt::T( 'Greater Than or Equal To' );
				break;
			case 5:
				$operator	=	CBTxt::T( 'Less Than or Equal To' );
				break;
			case 6:
				$operator	=	CBTxt::T( 'Empty' );
				break;
			case 7:
				$operator	=	CBTxt::T( 'Not Empty' );
				break;
			case 8:
				$operator	=	CBTxt::T( 'Does Contain' );
				break;
			case 9:
				$operator	=	CBTxt::T( 'Does Not Contain' );
				break;
			case 10:
				$operator	=	CBTxt::T( 'Is REGEX' );
				break;
			case 11:
				$operator	=	CBTxt::T( 'Is Not REGEX' );
				break;
			case 12:
				$operator	=	CBTxt::T( 'Is In' );
				break;
			case 13:
				$operator	=	CBTxt::T( 'Is Not In' );
				break;
			case 0:
			default:
				$operator	=	CBTxt::T( 'Equal To' );
				break;
		}

		return $operator;
	}

	/**
	 * Checks if the field is empty based off fieldtype
	 *
	 * @param FieldTable $field
	 * @param mixed      $value
	 * @return bool
	 */
	public static function isFieldEmpty( $field, $value )
	{
		if ( ( $value === '' ) || ( $value === null ) ) {
			return true;
		}

		switch ( $field->getString( 'type' ) ) {
			case 'fieldgroup':
				return ( $value === '[]' );
			case 'date':
				return ( $value === '0000-00-00' );
			case 'datetime':
				return ( $value === '0000-00-00 00:00:00' );
			case 'time':
				return ( $value === '00:00:00' );
			case 'integer':
			case 'points':
			case 'checkbox':
			case 'terms':
			case 'counter':
				return ( (int) $value === 0 );
			case 'rating':
			case 'float':
				return ( (float) $value === 0.0 );
		}

		return false;
	}

	/**
	 * Grabs field value from POST or user object based off location
	 *
	 * @param int|UserTable $user
	 * @param FieldTable    $field
	 * @param string        $reason
	 * @param bool          $reset
	 * @return array|mixed|string
	 */
	private static function getFieldValue( $user, $field, $reason, $reset )
	{
		global $_PLUGINS, $_CB_fieldIconDisplayed;

		static $values											=	array();

		$fieldId												=	$field->getInt( 'fieldid', 0 );

		if ( ! $fieldId ) {
			return null;
		}

		if ( ! $user instanceof UserTable ) {
			$user												=	\CBuser::getUserDataInstance( (int) $user );
		}

		$post													=	Application::Input()->getNamespaceRegistry( 'post' );
		$checkPost												=	false;

		if ( in_array( $reason, array( 'register', 'edit' ), true ) && $post->count() ) {
			$view												=	Application::Input()->getString( 'view' );

			if ( Application::Application()->isClient( 'administrator' ) && in_array( $view, array( 'apply', 'save' ), true ) ) {
				$checkPost										=	true;
			} elseif ( in_array( $view, array( 'saveregisters', 'saveuseredit' ), true ) ) {
				$checkPost										=	true;
			}

			if ( $checkPost ) {
				if ( $field->getInt( 'readonly', 0 ) && ( $reason !== 'register' ) && ( ! Application::Application()->isClient( 'administrator' ) ) ) {
					$checkPost									=	false;
				} elseif ( ( $reason === 'register' ) && ( ! $field->getInt( 'registration', 1 ) ) ) {
					$checkPost									=	false;
				} elseif ( ( $reason === 'edit' ) && ( ! $field->getInt( 'edit', 1 ) ) ) {
					$checkPost									=	false;
				}
			}
		}

		$userId													=	$user->getInt( 'id', 0 );
		$fieldName												=	$field->getString( 'name' );

		if ( ! isset( $values[$fieldName][$userId][$reason][$checkPost] ) ) {
			if ( ! $field->getRaw( 'params' ) instanceof ParamsInterface ) {
				$field->set( 'params', new Registry( $field->getRaw( 'params' ) ) );
			}

			$fieldValue											=	null;

			if ( $checkPost ) {
				$field->set( '_noCondition', true );

				$iconsRendered									=	isset( $_CB_fieldIconDisplayed[$fieldId] );
				$canEdit										=	$_PLUGINS->callField( $field->getString( 'type' ), 'getFieldRow', array( &$field, &$user, 'htmledit', 'none', $reason, 0 ), $field );

				if ( ( ! $iconsRendered ) && isset( $_CB_fieldIconDisplayed[$fieldId] ) ) {
					unset( $_CB_fieldIconDisplayed[$fieldId] );
				}

				$field->set( '_noCondition', false );

				if ( trim( $canEdit ) !== '' ) {
					$postUser									=	new UserTable();

					foreach ( array_keys( get_object_vars( $user ) ) as $k ) {
						if ( strpos( $k, '_' ) !== 0 ) {
							$postUser->set( $k, $user->getRaw( $k ) );
						}
					}

					if ( ! $post->has( $fieldName ) ) {
						if ( $reset ) {
							self::resetFieldValue( $post, $field );
						} else {
							$post->set( $fieldName, null );
						}
					}

					$postUser->bindThisUserFromDbArray( $post->asArray() );

					$fieldValue									=	$postUser->getRaw( $fieldName );

					if ( is_array( $fieldValue ) ) {
						$fieldValue								=	implode( '|*|', $fieldValue );
					}

					if ( $fieldValue === null ) {
						$field->set( '_noCondition', true );

						$fieldValue								=	$_PLUGINS->callField( $field->getString( 'type' ), 'getFieldRow', array( &$field, &$postUser, 'php', 'none', 'profile', 0 ), $field );

						$field->set( '_noCondition', false );

						if ( is_array( $fieldValue ) ) {
							$fieldValue							=	array_shift( $fieldValue );

							if ( is_array( $fieldValue ) ) {
								$fieldValue						=	implode( '|*|', $fieldValue );
							}
						}
					}
				}
			}

			if ( $fieldValue === null ) {
				$fieldValue										=	$user->getRaw( $fieldName );

				if ( is_array( $fieldValue ) ) {
					$fieldValue									=	implode( '|*|', $fieldValue );
				}

				if ( $fieldValue === null ) {
					$field->set( '_noCondition', true );

					$fieldValue									=	$_PLUGINS->callField( $field->getString( 'type' ), 'getFieldRow', array( &$field, &$user, 'php', 'none', 'profile', 0 ), $field );

					$field->set( '_noCondition', false );

					if ( is_array( $fieldValue ) ) {
						$fieldValue								=	array_shift( $fieldValue );

						if ( is_array( $fieldValue ) ) {
							$fieldValue							=	implode( '|*|', $fieldValue );
						}
					}
				}
			}

			$values[$fieldName][$userId][$reason][$checkPost]	=	$fieldValue;
		}

		return $values[$fieldName][$userId][$reason][$checkPost];
	}

	/**
	 * Parses substitution extras array from available variables
	 *
	 * @return array
	 */
	private static function getSubstitutions()
	{
		static $extras		=	array();

		if ( empty( $extras ) ) {
			$input			=	Application::Input();

			$get			=	$input->getNamespaceRegistry( 'get' );

			if ( $get ) {
				self::prepareExtras( 'get', $get->asArray(), $extras );
			}

			$post			=	$input->getNamespaceRegistry( 'post' );

			if ( $post ) {
				self::prepareExtras( 'post', $post->asArray(), $extras );
			}

			$files			=	$input->getNamespaceRegistry( 'files' );

			if ( $files ) {
				self::prepareExtras( 'files', $files->asArray(), $extras );
			}

			$cookie			=	$input->getNamespaceRegistry( 'cookie' );

			if ( $cookie ) {
				self::prepareExtras( 'cookie', $cookie->asArray(), $extras );
			}

			$server			=	$input->getNamespaceRegistry( 'server' );

			if ( $server ) {
				self::prepareExtras( 'server', $server->asArray(), $extras );
			}

			$env			=	$input->getNamespaceRegistry( 'env' );

			if ( $env ) {
				self::prepareExtras( 'env', $env->asArray(), $extras );
			}
		}

		return $extras;
	}

	/**
	 * Converts array or object into pathed extras substitutions
	 *
	 * @param string       $prefix
	 * @param array|object $items
	 * @param array        $extras
	 */
	private static function prepareExtras( $prefix, $items, &$extras )
	{
		foreach ( $items as $k => $v ) {
			if ( is_array( $v ) ) {
				$multi					=	false;

				foreach ( $v as $kv => $cv ) {
					if ( is_numeric( $kv ) ) {
						$kv				=	(int) $kv;
					}

					if ( is_object( $cv ) || is_array( $cv ) || ( $kv && ( ! is_int( $kv ) ) ) ) {
						$multi			=	true;
					}
				}

				if ( ! $multi ) {
					$v					=	implode( '|*|', $v );
				}
			}

			$k							=	'_' . ltrim( str_replace( ' ', '_', strtolower( trim( $k ) ) ), '_' );

			if ( ( ! is_object( $v ) ) && ( ! is_array( $v ) ) ) {
				$extras[$prefix . $k]	=	$v;
			} elseif ( $v ) {
				if ( is_object( $v ) ) {
					/** @var object $v */
					$subItems			=	get_object_vars( $v );
				} else {
					$subItems			=	$v;
				}

				self::prepareExtras( $prefix . $k, $subItems, $extras );
			}
		}
	}

	/**
	 * @return array
	 */
	public static function loadFields()
	{
 		global $_CB_database;

		static $values		=	null;

		if ( $values === null ) {
			$values			=	array();

			$query			=	"SELECT f." . $_CB_database->NameQuote( 'fieldid' ) . " AS value"
							.	", f." . $_CB_database->NameQuote( 'title' ) . " AS text"
							.	", f." . $_CB_database->NameQuote( 'name' )
							.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_fields' ) . " AS f"
							.	"\n LEFT JOIN " . $_CB_database->NameQuote( '#__comprofiler_tabs' ) . " AS t"
							.	" ON t." . $_CB_database->NameQuote( 'tabid' ) . " = f." . $_CB_database->NameQuote( 'tabid' )
							.	"\n WHERE f." . $_CB_database->NameQuote( 'published' ) . " = 1"
							.	"\n AND f." . $_CB_database->NameQuote( 'name' ) . " != " . $_CB_database->Quote( 'NA' )
							.	"\n ORDER BY t." . $_CB_database->NameQuote( 'position' ) . ", t." . $_CB_database->NameQuote( 'ordering' ) . ", f." . $_CB_database->NameQuote( 'ordering' );
			$_CB_database->setQuery( $query );
			$fields			=	$_CB_database->loadObjectList();

			foreach ( $fields as $field ) {
				$values[]	=	\moscomprofilerHTML::makeOption( (string) $field->value, CBTxt::T( $field->text ) . ' (' . $field->name . ')' );
			}
		}

		return $values;
	}

	/**
	 * @param UserTable|ParamsInterface $object
	 * @param FieldTable                $field
	 */
	public static function resetFieldValue( $object, $field )
	{
		$isUser		=	( $object instanceof UserTable );
		$fieldName	=	$field->getString( 'name' );

		switch ( $field->getString( 'type' ) ) {
			case 'fieldgroup':
				if ( $isUser && ( ! isset( $object->$fieldName ) ) ) {
					return;
				}

				$object->set( $fieldName, '[]' );
				break;
			case 'date':
				if ( $isUser && ( ! isset( $object->$fieldName ) ) ) {
					return;
				}

				$object->set( $fieldName, '0000-00-00' );
				break;
			case 'datetime':
				if ( $isUser && ( ! isset( $object->$fieldName ) ) ) {
					return;
				}

				$object->set( $fieldName, '0000-00-00 00:00:00' );
				break;
			case 'time':
				if ( isset( $object->$fieldName ) ) {
					$object->set( $fieldName, '00:00:00' );
				}
				break;
			case 'integer':
			case 'float':
			case 'points':
			case 'rating':
			case 'checkbox':
			case 'terms':
			case 'counter':
				if ( $isUser && ( ! isset( $object->$fieldName ) ) ) {
					return;
				}

				$object->set( $fieldName, 0 );
				break;
			case 'image':
				if ( $isUser && ( ! isset( $object->$fieldName ) ) ) {
					return;
				}

				$object->set( $fieldName, '' );

				$approvedName	=	$fieldName . 'approved';

				if ( $isUser && ( ! isset( $object->$approvedName ) ) ) {
					return;
				}

				$object->set( $approvedName, 0 );
				break;
			default:
				foreach ( $field->getTableColumns() as $column ) {
					if ( $isUser && ( ! isset( $object->$column ) ) ) {
						continue;
					}

					$object->set( $column, '' );
				}
				break;
		}
	}

	/**
	 * @return bool
	 */
	public static function checkCBSubsExists()
	{
		global $_PLUGINS;

		if ( $_PLUGINS->getLoadedPlugin( 'user', 'cbpaidsubscriptions' ) ) {
			return true;
		}

		return false;
	}

	/**
	 * @return array
	 */
	public static function getCBSubsPlans()
	{
		if ( ! self::checkCBSubsExists() ) {
			return array();
		}

		$plansMgr				=	\cbpaidPlansMgr::getInstance();
		$plans					=	$plansMgr->loadPublishedPlans( null, true, 'any', null );
		$plansList				=	array();

		if ( $plans ) {
			$plansList			=	array();

			foreach ( $plans as $k => $plan ) {
				$plansList[]	=	\moscomprofilerHTML::makeOption( (string) $k, $plan->getString( 'alias' ) );
			}
		}

		return $plansList;
	}

	/**
	 * @param UserTable   $user
	 * @param null|string $reason
	 * @return array
	 */
	private static function getCBSubsSubscriptionsStatus( $user, $reason = null )
	{
		global $_CB_framework;

		if ( ! self::checkCBSubsExists() ) {
			return array();
		}

		static $cache								=	array();

		$userId										=	$user->getInt( 'id', 0 );

		if (  ! isset( $cache[$userId] ) ) {
			$cache[$userId]							=	array();

			// Check for selected plans:
			if ( $reason === 'register' ) {
				$subscriptionsGUI					=	new \cbpaidControllerUI();
				$chosenPlans						=	$subscriptionsGUI->getAndCheckChosenRegistrationPlans( $_POST, $user );

				if ( is_array( $chosenPlans ) && ( count( $chosenPlans ) > 0 ) ) {
					foreach ( $chosenPlans as $chosenPlan ) {
						$planId						=	$chosenPlan->getInt( 'id', 0 );

						$cache[$userId][$planId]	=	'I';
					}
				}
			} elseif ( ( $reason === 'edit' ) && Application::Application()->isClient( 'administrator' ) ) {
				$subscriptionsGUI					=	new \cbpaidControllerUI();
				$chosenPlans						=	$subscriptionsGUI->getAndCheckChosenUpgradePlans( $_POST, $user, $_CB_framework->now() );

				if ( is_array( $chosenPlans ) && ( count( $chosenPlans ) > 0 ) ) {
					foreach ( $chosenPlans as $chosenPlan ) {
						$planId						=	$chosenPlan->getInt( 'id', 0 );

						$cache[$userId][$planId]	=	'I';
					}
				}
			}

			// Now check for actual subscriptions:
			if ( $userId ) {
				$paidUserExtension					=	\cbpaidUserExtension::getInstance( $userId );
				$subscriptions						=	$paidUserExtension->getUserSubscriptions();

				foreach ( $subscriptions as $subscription ) {
					$planId							=	$subscription->getInt( 'plan_id', 0 );

					$cache[$userId][$planId]		=	$subscription->getString( 'status' );
				}
			}
		}

		return $cache[$userId];
	}

	/**
	 * Checks if create_function is available
	 *
	 * @return bool
	 */
	private static function canCreateFunction()
	{
		static $cache					=	null;

		if ( $cache === null ) {
			$cache						=	function_exists( 'create_function' );

			if ( $cache && function_exists( 'ini_get' ) ) {
				$disabledFunctions		=	ini_get( 'disabled_functions' );

				if ( $disabledFunctions !== false ) {
					$cache				=	( ! in_array( 'create_function', explode( ',', $disabledFunctions ), true ) );
				}
			}

			if ( $cache && ( PHP_VERSION_ID >= 70200 ) ) {
				$cache					=	false;
			}
		}

		return $cache;
	}

	/**
	 * Checks if eval is available
	 *
	 * @return bool
	 */
	private static function canEval()
	{
		static $cache					=	null;

		if ( $cache === null ) {
			$cache						=	true;

			if ( function_exists( 'ini_get' ) ) {
				$disabledFunctions		=	ini_get( 'disabled_functions' );

				if ( $disabledFunctions !== false ) {
					$cache				=	( ! in_array( 'eval', explode( ',', $disabledFunctions ), true ) );
				}

				if ( $cache ) {
					$evalDisabled		=	ini_get( 'suhosin.executor.disable_eval' );

					if ( ( $evalDisabled === true ) || ( $evalDisabled === '1' ) ) {
						$cache			=	false;
					}
				}
			}
		}

		return $cache;
	}

	/**
	 * Attempts to execute code from a string
	 *
	 * @param string          $code
	 * @param FieldTable|null $field
	 * @param TabTable|null   $tab
	 * @param UserTable|null  $user
	 * @return string|null
	 */
	private static function outputCode( $code, $field = null, $tab = null, $user = null )
	{
		if ( ! $code ) {
			return null;
		}

		ob_start();
		if ( self::canCreateFunction() ) {
			// PHP 7 has a catchable error so lets catch it if we can and throw it as exception to be caught later:
			if ( PHP_VERSION_ID >= 70000 ) {
				try {
					$function			=	create_function( '$field,$tab,$user', $code );
					$return				=	$function( $field, $tab, $user );
				} catch ( \Error $e ) {
					throw new \Exception( $e->getMessage() );
				}
			} else {
				$function				=	create_function( '$field,$tab,$user', $code );
				$return					=	$function( $field, $tab, $user );
			}
		} elseif ( self::canEval() ) {
			$function					=	'$function = function( $field, $tab, $user ) { ' . $code . ' };';

			// PHP 7 has a catchable error so lets catch it if we can and throw it as exception to be caught later:
			if ( PHP_VERSION_ID >= 70000 ) {
				try {
					eval( $function );

					$return				=	$function( $field, $tab, $user );
				} catch ( \Error $e ) {
					throw new \Exception( $e->getMessage() );
				}
			} else {
				eval( $function );

				$return					=	$function( $field, $tab, $user );
			}
		} else {
			// Last resort fallback; this is slow so hopefully it doesn't come to this:
			$temp						=	tmpfile();

			fwrite( $temp, "<?php \n" . $code );

			$tempData					=	stream_get_meta_data( $temp );

			// PHP 7 has a catchable error so lets catch it if we can and throw it as exception to be caught later:
			if ( PHP_VERSION_ID >= 70000 ) {
				try {
					$return				=	include $tempData['uri'];
				} catch ( \Error $e ) {
					fclose( $temp );

					throw new \Exception( $e->getMessage() );
				}
			} else {
				$return					=	include $tempData['uri'];
			}

			fclose( $temp );

			if ( ! is_string( $return ) ) {
				$return					=	null;
			}
		}
		$echos							=	ob_get_clean();

		if ( $echos && ( is_string( $return ) || ( $return === '' ) || ( $return === null ) ) ) {
			$return						=	$echos . $return;
		}

		return $return;
	}

	/**
	 * Encodes a string to SQL safe
	 *
	 * @param string $str
	 * @return string
	 */
	public static function escapeSQL( $str )
	{
		global $_CB_database;

		return $_CB_database->getEscaped( $str );
	}
}
