<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\Invites;

use CB\Database\Table\UserTable;
use CB\Plugin\Invites\Table\InviteTable;
use CBLib\Application\Application;
use CBLib\Registry\Registry;
use CBLib\Language\CBTxt;

defined('CBLIB') or die();

class CBInvites
{

	/**
	 * @return Registry
	 */
	public static function getGlobalParams()
	{
		global $_PLUGINS;

		static $params	=	null;

		if ( ! $params ) {
			$plugin		=	$_PLUGINS->getLoadedPlugin( 'user', 'cbinvites' );
			$params		=	new Registry();

			if ( $plugin ) {
				$params->load( $plugin->params );
			}
		}

		return $params;
	}

	/**
	 * @param null|string $template
	 * @param null|string $file
	 * @param bool|array  $headers
	 * @return null|string
	 */
	public static function getTemplate( $template = null, $file = null, $headers = array( 'template', 'override' ) )
	{
		global $_CB_framework, $_PLUGINS;

		if ( $headers === true ) {
			$headers					=	array( 'template', 'override' );
		}

		$plugin							=	$_PLUGINS->getLoadedPlugin( 'user', 'cbinvites' );

		if ( ! $plugin ) {
			return null;
		}

		static $defaultTemplate			=	null;

		if ( $defaultTemplate === null ) {
			$defaultTemplate			=	self::getGlobalParams()->getString( 'general_template', 'default' );
		}

		if ( ( $template === '' ) || ( $template === null ) || ( $template === '-1' ) ) {
			$template					=	$defaultTemplate;
		}

		if ( ! $template ) {
			$template					=	'default';
		}

		$livePath						=	$_PLUGINS->getPluginLivePath( $plugin );
		$absPath						=	$_PLUGINS->getPluginPath( $plugin );

		if ( $headers !== false ) {
			static $loaded				=	array();

			$loaded[$template]			=	array();

			// Global CSS File:
			if ( in_array( 'template', $headers ) && ( ! in_array( 'template', $loaded[$template] ) ) ) {
				$global					=	'/templates/' . $template . '/template.css';

				if ( ! file_exists( $absPath . $global ) ) {
					$global				=	'/templates/default/template.css';
				}

				if ( file_exists( $absPath . $global ) ) {
					$_CB_framework->document->addHeadStyleSheet( $livePath . $global );
				}

				$loaded[$template][]	=	'template';
			}

			// File or Custom CSS/JS Headers:
			foreach ( $headers as $header ) {
				if ( in_array( $header, $loaded[$template] ) || in_array( $header, array( 'template', 'override' ) ) ) {
					continue;
				}

				$header					=	preg_replace( '/[^-a-zA-Z0-9_]/', '', $header );

				if ( ! $header ) {
					continue;
				}

				$css					=	'/templates/' . $template . '/' . $header . '.css';
				$js						=	'/templates/' . $template . '/' . $header . '.js';

				if ( ! file_exists( $absPath . $css ) ) {
					$css				=	'/templates/default/' . $header . '.css';
				}

				if ( file_exists( $absPath . $css ) ) {
					$_CB_framework->document->addHeadStyleSheet( $livePath . $css );
				}

				if ( ! file_exists( $absPath . $js ) ) {
					$js					=	'/templates/default/' . $header . '.js';
				}

				if ( file_exists( $absPath . $js ) ) {
					$_CB_framework->document->addHeadScriptUrl( $livePath . $js );
				}

				$loaded[$template][]	=	$header;
			}

			// Override CSS File:
			if ( in_array( 'override', $headers ) && ( ! in_array( 'override', $loaded[$template] ) ) ) {
				$override				=	'/templates/' . $template . '/override.css';

				if ( file_exists( $absPath . $override ) ) {
					$_CB_framework->document->addHeadStyleSheet( $livePath . $override );
				}

				$loaded[$template][]	=	'override';
			}
		}

		$path							=	array();

		foreach ( explode( '/', $file ) as $part ) {
			$path[]						=	preg_replace( '/[^-a-zA-Z0-9_]/', '', $part );
		}

		$file							=	implode( '/', $path );
		$return							=	null;

		if ( $file ) {
			if ( $headers !== false ) {
				$headers[]				=	$file;
			}

			$php						=	$absPath . '/templates/' . $template . '/' . $file . '.php';

			if ( ! file_exists( $php ) ) {
				$php					=	$absPath . '/templates/default/' . $file . '.php';
			}

			if ( file_exists( $php ) ) {
				$return					=	$php;
			}
		}

		return $return;
	}

	/**
	 * Returns the current return url or generates one from current page
	 *
	 * @param bool|false $current
	 * @param bool|false $raw
	 * @return null|string
	 */
	public static function getReturn( $current = false, $raw = false )
	{
		static $cache				=	array();

		if ( ! isset( $cache[$current] ) ) {
			$url					=	null;

			if ( $current ) {
				$returnUrl			=	Application::Input()->getBase64( 'get/return' );

				if ( $returnUrl ) {
					$returnUrl		=	base64_decode( $returnUrl );

					if ( \JUri::isInternal( $returnUrl ) || ( $returnUrl[0] === '/' ) ) {
						$url		=	$returnUrl;
					}
				}
			} else {
				$isHttps			=	( isset( $_SERVER['HTTPS'] ) && ( ! empty( $_SERVER['HTTPS'] ) ) && ( $_SERVER['HTTPS'] !== 'off' ) );
				$returnUrl			=	'http' . ( $isHttps ? 's' : '' ) . '://' . $_SERVER['HTTP_HOST'];

				if ( ( ! empty( $_SERVER['PHP_SELF'] ) ) && ( ! empty( $_SERVER['REQUEST_URI'] ) ) ) {
					$returnUrl		.=	$_SERVER['REQUEST_URI'];
				} else {
					$returnUrl		.=	$_SERVER['SCRIPT_NAME'];

					if ( isset( $_SERVER['QUERY_STRING'] ) && ( ! empty( $_SERVER['QUERY_STRING'] ) ) ) {
						$returnUrl	.=	'?' . $_SERVER['QUERY_STRING'];
					}
				}

				$url				=	cbUnHtmlspecialchars( preg_replace( '/[\\\"\\\'][\\s]*javascript:(.*)[\\\"\\\']/', '""', preg_replace( '/eval\((.*)\)/', '', htmlspecialchars( urldecode( $returnUrl ) ) ) ) );
			}

			$cache[$current]		=	$url;
		}

		$return						=	$cache[$current];

		if ( ( ! $raw ) && $return ) {
			$return					=	base64_encode( $return );
		}

		return $return;
	}

	/**
	 * Redirects to the return url if available otherwise to the url specified
	 *
	 * @param string      $url
	 * @param null|string $message
	 * @param string      $messageType
	 */
	public static function returnRedirect( $url, $message = null, $messageType = 'message' )
	{
		$returnUrl		=	self::getReturn( true, true );

		cbRedirect( ( $returnUrl ? $returnUrl : $url ), $message, $messageType );
	}

	/**
	 * @param InviteTable $row
	 * @return bool
	 */
	public static function sendInvite( &$row )
	{
		global $_CB_framework;

		$params					=	self::getGlobalParams();
		$cbUser					=	\CBuser::getInstance( $row->getInt( 'user_id', 0 ), false );

		$extra					=	array(	'site'		=>	$_CB_framework->getCfg( 'live_site' ),
											'sitename'	=>	$_CB_framework->getCfg( 'sitename' ),
											'sitedesc'	=>	$_CB_framework->getCfg( 'MetaDesc' ),
											'register'	=>	$_CB_framework->viewUrl( 'registers' ),
											'profile'	=>	$_CB_framework->viewUrl( 'userprofile', true, array( 'user' => $row->getInt( 'user_id', 0 ) ) ),
											'invite'	=>	$row->getLink(),
											'code'		=>	$row->getString( 'code' ),
											'to'		=>	$row->getString( 'to' )
										);

		$mailFromName			=	$cbUser->replaceUserVars( $params->getString( 'invite_from_name' ), true, false, $extra );
		$mailFromAddr			=	$cbUser->replaceUserVars( $params->getString( 'invite_from_address' ), true, false, $extra );
		$mailSubject			=	$cbUser->replaceUserVars( CBTxt::T( $params->getString( 'invite_subject', 'You have been invited by [formatname] to join us!' ) ), true, false, $extra );
		$mailBody				=	$cbUser->replaceUserVars( CBTxt::T( $params->getHtml( 'invite_body', 'Hello!<p><a href="[profile]">[formatname]</a> would like to invite you to join <a href="[site]">[sitename]</a>! Please use the following invite link if you would like to join.</p><p><a href="[invite]">[invite]</a></p>' ) ), false, false, $extra );
		$mailTo					=	$row->getString( 'to' );
		$mailCC					=	$cbUser->replaceUserVars( $params->getString( 'invite_cc' ), true, false, $extra );
		$mailBCC				=	$cbUser->replaceUserVars( $params->getString( 'invite_bcc' ), true, false, $extra );
		$mailAttachments		=	$cbUser->replaceUserVars( $params->getString( 'invite_attachments' ), true, false, $extra );

		if ( $mailCC ) {
			$mailCC				=	preg_split( '/ *, */', $mailCC );
		}

		if ( $mailBCC ) {
			$mailBCC			=	preg_split( '/ *, */', $mailBCC );
		}

		if ( $mailAttachments ) {
			$mailAttachments	=	preg_split( '/ *, */', $mailAttachments );
		}

		if ( $mailTo && $mailSubject && $mailBody ) {
			$notification		=	new \cbNotification();

			if ( $notification->sendFromSystem( $mailTo, $mailSubject, $mailBody, false, 1, $mailCC, $mailBCC, $mailAttachments, $extra, true, $mailFromName, $mailFromAddr ) ) {
				return true;
			}

			$row->setError( ( $notification->errorMSG ? $notification->errorMSG : CBTxt::T( 'Email failed to send.' ) ) );
		} elseif ( ! $mailTo ) {
			$row->setError( CBTxt::T( 'Email address missing.' ) );
		} elseif ( ! $mailSubject ) {
			$row->setError( CBTxt::T( 'Email subject missing.' ) );
		} elseif ( ! $mailBody ) {
			$row->setError( CBTxt::T( 'Email body missing.' ) );
		}

		return false;
	}

	/**
	 * @param UserTable $user
	 * @param string    $type
	 * @return bool
	 */
	public static function canCreateInvite( $user, $type = 'emails' )
	{
		if ( ! $user->getInt( 'id', 0 ) ) {
			return true;
		}

		if ( Application::MyUser()->isGlobalModerator() ) {
			return true;
		}

		if ( ! self::getGlobalParams()->getBool( 'invite_' . $type, true ) ) {
			return false;
		}

		if ( $user->getInt( 'id', 0 ) !== Application::MyUser()->getUserId() ) {
			return false;
		}

		if ( ! Application::User( $user->getInt( 'id', 0 ) )->canViewAccessLevel( self::getGlobalParams()->getInt( 'invite_create_access', 2 ) ) ) {
			return false;
		}

		if ( self::getCreateLimited( $user ) ) {
			return false;
		}

		return true;
	}

	/**
	 * @param UserTable $user
	 * @return bool
	 */
	public static function getCreateLimited( $user )
	{
		global $_CB_database;

		$inviteLimit	=	self::getCreateLimit( $user );

		if ( ! $inviteLimit ) {
			return false;
		}

		$query			=	'SELECT COUNT(*)'
						.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_invites' )
						.	"\n WHERE " . $_CB_database->NameQuote( 'user_id' ) . " = " . $user->getInt( 'id', 0 )
						.	"\n AND ( " . $_CB_database->NameQuote( 'user' ) . " IS NULL OR " . $_CB_database->NameQuote( 'user' ) . " = " . $_CB_database->Quote( '' ) . " )";
		$_CB_database->setQuery( $query );
		return ( (int) $_CB_database->loadResult() >= $inviteLimit );
	}

	/**
	 * @param UserTable $user
	 * @return int
	 */
	public static function getCreateLimit( $user )
	{
		$createLimit			=	self::getGlobalParams()->getString( 'invite_limit', 'custom' );

		if ( is_numeric( $createLimit ) ) {
			// covers B/C case where this used to be an integer:
			return (int) $createLimit;
		}

		if ( $createLimit !== 'custom' ) {
			$limitField			=	\CBuser::getInstance( $user->getInt( 'id', 0 ), false )->getField( $createLimit, null, 'php', 'none', 'profile', 0, true );

			if ( is_array( $limitField ) ) {
				$limitField		=	array_shift( $limitField );

				if ( is_array( $limitField ) ) {
					$limitField	=	implode( '|*|', $limitField );
				}
			} else {
				$limitField		=	$user->getInt( $createLimit, 0 );
			}

			return (int) $limitField;
		}

		return self::getGlobalParams()->getInt( 'invite_limit_custom', 0 );
	}

	/**
	 * @return bool
	 */
	public static function isHTTPS()
	{
		global $_CB_framework;

		static $cache	=	null;

		if ( $cache === null ) {
			$cache		=	false;

			if ( isset( $_SERVER['HTTPS'] ) && ( ! empty( $_SERVER['HTTPS'] ) ) && ( $_SERVER['HTTPS'] !== 'off' ) ) {
				$cache	=	true;
			} elseif ( parse_url( $_CB_framework->getCfg( 'live_site' ), PHP_URL_HOST ) === 'localhost' ) {
				$cache	=	true;
			}
		}

		return $cache;
	}

	/**
	 * Outputs copy input for invite link
	 * Note this is strictly for backend XML usage only!
	 *
	 * @param string        $value
	 * @param null|Registry $pluginParams
	 * @param string        $name
	 * @param string        $node
	 * @param string        $control_name
	 * @param string        $control_name_name
	 * @param bool          $view
	 * @param Registry      $modelOfData
	 * @return null|string
	 */
	public static function getCopyLink( $value, $pluginParams, $name, $node, $control_name, $control_name_name, $view, $modelOfData )
	{
		global $_CB_framework;

		if ( method_exists( $modelOfData, 'getStorage' ) ) {
			$invite		=	$modelOfData->getStorage();
		} else {
			$invite		=	$modelOfData;
		}

		if ( ( ! $invite ) && ( ! ( $invite instanceof InviteTable ) ) ) {
			return null;
		}

		static $loaded	=	0;

		if ( ! $loaded++ ) {
			$js			=	"$( '.invitesInviteLinkCopy' ).on( 'click', function() {"
						.		"$( this ).closest( '.invitesInviteLink' ).find( '.invitesInviteLinkInput' ).click();"
						.	"});";

			$_CB_framework->outputCbJQuery( $js );
		}

		return		'<div class="input-group flex-nowrap invitesInviteLink">'
				.		'<input type="text" value="' . $invite->getLink() . '" class="form-control form-control-sm rounded-0 invitesInviteLinkInput" style="cursor: pointer;" onclick="this.select(); document.execCommand( \'copy\' );" readonly aria-label="' . htmlspecialchars( CBTxt::T( 'Invite Link' ) ) . '" data-hascbtooltip="true" data-cbtooltip-tooltip="' . htmlspecialchars( CBTxt::T( 'Copied!' ) ) . '" data-cbtooltip-simple="true" data-cbtooltip-open-event="click" data-cbtooltip-close-event="unfocus" data-cbtooltip-button-close="false" />'
				.		'<div class="input-group-append">'
				.			'<button type="button" class="rounded-0 btn btn-sm btn-secondary invitesInviteLinkCopy">' . CBTxt::T( 'Copy' ) . '</button>'
				.		'</div>'
				.	'</div>';
	}
}
