<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\AutoActions\Action;

use CB\Database\Table\UserTable;
use CB\Plugin\Privacy\Privacy;
use CB\Plugin\Privacy\Table\PrivacyTable;
use CBLib\Language\CBTxt;
use CBLib\Registry\ParamsInterface;

\defined( 'CBLIB' ) or die();

class PrivacyAction extends Action
{

	/**
	 * @param UserTable $user
	 * @return null|string|bool
	 */
	public function execute( UserTable $user )
	{
		global $_CB_database;

		if ( ! $this->installed() ) {
			$this->error( CBTxt::T( 'AUTO_ACTION_PRIVACY_NOT_INSTALLED', ':: Action [action] :: CB Privacy is not installed', [ '[action]' => $this->autoaction()->getId() ] ) );
			return null;
		}

		$return								=	null;

		foreach ( $this->autoaction()->getParams()->subTree( 'privacy' ) as $row ) {
			/** @var ParamsInterface $row */
			$owner							=	$row->getString( 'owner', '' );

			if ( ! $owner ) {
				$owner						=	$user->getInt( 'id', 0 );
			} else {
				$owner						=	(int) $this->string( $user, $owner );
			}

			if ( ! $owner ) {
				$this->error( CBTxt::T( 'AUTO_ACTION_PRIVACY_NO_OWNER', ':: Action [action] :: CB Privacy skipped due to missing owner', [ '[action]' => $this->autoaction()->getId() ] ) );
				continue;
			}

			if ( $user->getInt( 'id', 0 ) !== $owner ) {
				$actionUser					=	\CBuser::getUserDataInstance( $owner );
			} else {
				$actionUser					=	$user;
			}

			$asset							=	$this->string( $actionUser, $row->getString( 'asset', '' ) );

			if ( ! $asset ) {
				$this->error( CBTxt::T( 'AUTO_ACTION_PRIVACY_NO_ASSET', ':: Action [action] :: CB Privacy skipped due to missing asset', [ '[action]' => $this->autoaction()->getId() ] ) );
				continue;
			}

			$mode							=	$row->getString( 'mode', '' );

			if ( $mode === 'privacy' ) {
				$method						=	$row->getString( 'method', 'display' );

				$privacy					=	new Privacy( $asset, $actionUser );

				$privacy->set( 'autoaction', $this->autoaction()->getId() );

				$privacy->parse( $row->subTree( 'privacy_privacy' ), 'privacy_' );

				if ( $method === 'authorized' ) {
					$return					=	$privacy->authorized();

					if ( ! $return ) {
						$loop				=	$this->autoaction()->getParams()->getInt( 'loop', 0 );

						if ( $loop ) {
							$variables		=	$this->variables();
							$loopKey		=	(int) $variables['loop_key'];

							if ( isset( $variables['var' . $loop][$loopKey] ) ) {
								unset( $variables['var' . $loop][$loopKey] );
							}
						}
					}
				} elseif ( $method === 'save' ) {
					$privacy->privacy( 'save' );
				} else {
					$return					.=	$privacy->privacy( 'edit' );
				}
			} else {
				$method						=	$row->getString( 'method', 'create' );

				if ( $method === 'delete' ) {
					$where					=	[];

					switch ( $row->getString( 'asset_owner', 'asset' ) ) {
						case 'owner':
							$where[]		=	$_CB_database->NameQuote( 'user_id' ) . ' = ' . $actionUser->getInt( 'id', 0 );
							break;
						case 'asset_owner':
							$where[]		=	$_CB_database->NameQuote( 'user_id' ) . ' = ' . $actionUser->getInt( 'id', 0 );
							$where[]		=	$_CB_database->NameQuote( 'asset' ) . ( strpos( $asset, '%' ) !== false ? ' LIKE ' : ' = ' ) . $_CB_database->Quote( $asset );
							break;
						case 'asset':
						default:
							$where[]		=	$_CB_database->NameQuote( 'asset' ) . ( strpos( $asset, '%' ) !== false ? ' LIKE ' : ' = ' ) . $_CB_database->Quote( $asset );
							break;
					}

					$query					=	'SELECT *'
											.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_privacy' )
											.	"\n WHERE " . implode( "\n AND ", $where );
					$_CB_database->setQuery( $query );
					$rules					=	$_CB_database->loadObjectList( 'id', '\CB\Plugin\Privacy\Table\PrivacyTable', [ $_CB_database ] );

					/** @var PrivacyTable[] $rules */
					foreach ( $rules as $rule ) {
						$rule->delete();
					}
				} else {
					$rules					=	explode( '|*|', $row->getString( 'rules', '' ) );

					if ( ! $rules ) {
						$this->error( CBTxt::T( 'AUTO_ACTION_PRIVACY_NO_RULES', ':: Action [action] :: CB Privacy skipped due to missing rules', [ '[action]' => $this->autoaction()->getId() ] ) );
						continue;
					}

					/** @var array $rules */
					foreach ( $rules as $rule ) {
						$privacy			=	new PrivacyTable();

						$privacy->load( [ 'user_id' => $actionUser->getInt( 'id', 0 ), 'asset' => $asset ] );

						$privacy->set( 'user_id', $actionUser->getInt( 'id', 0 ) );
						$privacy->set( 'asset', $asset );
						$privacy->set( 'rule', $rule );

						if ( ! $privacy->store() ) {
							$this->error( CBTxt::T( 'AUTO_ACTION_PRIVACY_FAILED', ':: Action [action] :: CB Privacy failed to save. Error: [error]', [ '[action]' => $this->autoaction()->getId(), '[error]' => $privacy->getError() ] ) );
						}
					}
				}
			}
		}

		return $return;
	}

	/**
	 * @return bool
	 */
	public function installed(): bool
	{
		global $_PLUGINS;

		$plugin				=	$_PLUGINS->getLoadedPlugin( 'user', 'cbprivacy' );

		if ( ! $plugin ) {
			return false;
		}

		$pluginVersion		=	str_replace( '+build.', '+', $_PLUGINS->getPluginVersion( $plugin, true ) );

		return version_compare( $pluginVersion, '5.0.0', '>=' ) && version_compare( $pluginVersion, '8.0.0', '<' );
	}
}