<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\AutoActions\Action;

use CB\Database\Table\UserTable;
use CB\Plugin\Activity\Activity;
use CB\Plugin\Activity\CBActivity;
use CB\Plugin\Activity\Comments;
use CB\Plugin\Activity\Following;
use CB\Plugin\Activity\Likes;
use CB\Plugin\Activity\Notifications;
use CB\Plugin\Activity\Table\ActivityTable;
use CB\Plugin\Activity\Table\CommentTable;
use CB\Plugin\Activity\Table\FollowTable;
use CB\Plugin\Activity\Table\HiddenTable;
use CB\Plugin\Activity\Table\LikeTable;
use CB\Plugin\Activity\Table\NotificationTable;
use CB\Plugin\Activity\Table\TagTable;
use CBLib\Application\Application;
use CBLib\Language\CBTxt;
use CBLib\Registry\ParamsInterface;

\defined( 'CBLIB' ) or die();

class ActivityAction extends Action
{

	/**
	 * @param UserTable $user
	 * @return string
	 */
	public function execute( UserTable $user ): string
	{
		global $_CB_database;

		if ( ! $this->installed() ) {
			$this->error( CBTxt::T( 'AUTO_ACTION_ACTIVITY_NOT_INSTALLED', ':: Action [action] :: CB Activity is not installed', [ '[action]' => $this->autoaction()->getId() ] ) );
			return '';
		}

		$return										=	'';

		foreach ( $this->autoaction()->getParams()->subTree( 'activity' ) as $row ) {
			/** @var ParamsInterface $row */
			$mode									=	$row->getString( 'mode', '' );
			$method									=	$row->getString( 'method', 'create' );
			$owner									=	$row->getString( 'owner', '' );

			if ( ! $owner ) {
				$owner								=	$user->getInt( 'id', 0 );
			} else {
				$owner								=	(int) $this->string( $user, $owner );
			}

			if ( $user->getInt( 'id', 0 ) !== $owner ) {
				$actionOwner						=	\CBuser::getUserDataInstance( $owner );
			} else {
				$actionOwner						=	$user;
			}

			$userId									=	$row->getString( 'user', '' );

			if ( ! $userId ) {
				$userId								=	$user->getInt( 'id', 0 );
			} else {
				$userId								=	(int) $this->string( $actionOwner, $userId );
			}

			if ( $user->getInt( 'id', 0 ) !== $userId ) {
				$actionUser							=	\CBuser::getUserDataInstance( $userId );
			} else {
				$actionUser							=	$user;
			}

			$asset									=	$this->notificationString( $userId, $actionOwner, $row->getString( 'asset', '' ) );

			if ( $mode === 'stream' ) {
				$stream								=	'';

				switch ( $row->getString( 'stream', 'activity' ) ) {
					case 'comments':
						if ( ! $asset ) {
							continue 2;
						}

						$commentsStream				=	new Comments( $asset, $actionOwner );

						$commentsStream->set( 'parent', $this->notificationString( $userId, $actionOwner, $row->getString( 'parent', '' ) ) );
						$commentsStream->set( 'moderators', explode( ',', $this->notificationString( $userId, $actionOwner, $row->getString( 'moderators', '' ) ) ) );
						$commentsStream->set( 'autoaction', $this->autoaction()->getId() );

						$commentsStream->parse( $row->subTree( 'comments_stream' ), 'comments_' );

						if ( ( ! CBActivity::canCreate( 'comment', $commentsStream ) ) && ( ! $commentsStream->rows() ) ) {
							continue 2;
						}

						if ( \in_array( $commentsStream->getString( 'layout', 'stream' ), [ 'button', 'toggle' ], true ) ) {
							$stream					=	$commentsStream->comments( 'button' );
						} else {
							$stream					=	$commentsStream->comments();
						}
						break;
					case 'follow':
						if ( ! $asset ) {
							continue 2;
						}

						$followingStream			=	new Following( $asset, $actionOwner );

						$followingStream->set( 'parent', $this->notificationString( $userId, $actionOwner, $row->getString( 'parent', '' ) ) );
						$followingStream->set( 'moderators', explode( ',', $this->notificationString( $userId, $actionOwner, $row->getString( 'moderators', '' ) ) ) );
						$followingStream->set( 'autoaction', $this->autoaction()->getId() );

						$followingStream->parse( $row->subTree( 'following_stream' ), 'following_' );

						if ( ( ! CBActivity::canCreate( 'follow', $followingStream ) ) && ( ! $followingStream->rows() ) ) {
							continue 2;
						}

						if ( $followingStream->getString( 'layout', 'button' ) === 'stream' ) {
							$stream					=	$followingStream->following();
						} else {
							$stream					=	$followingStream->following( 'button' );
						}
						break;
					case 'like':
						if ( ! $asset ) {
							continue 2;
						}

						$likesStream				=	new Likes( $asset, $actionOwner );

						$likesStream->set( 'parent', $this->notificationString( $userId, $actionOwner, $row->getString( 'parent', '' ) ) );
						$likesStream->set( 'moderators', explode( ',', $this->notificationString( $userId, $actionOwner, $row->getString( 'moderators', '' ) ) ) );
						$likesStream->set( 'autoaction', $this->autoaction()->getId() );

						$likesStream->parse( $row->subTree( 'likes_stream' ), 'likes_' );

						if ( ( ! CBActivity::canCreate( 'like', $likesStream ) ) && ( ! $likesStream->rows() ) ) {
							continue 2;
						}

						if ( $likesStream->getString( 'layout', 'button' ) === 'stream' ) {
							$stream					=	$likesStream->likes();
						} else {
							$stream					=	$likesStream->likes( 'button' );
						}
						break;
					case 'notifications':
						if ( ( ! $actionOwner->getInt( 'id', 0 ) ) || ( $actionOwner->getInt( 'id', 0 ) !== Application::MyUser()->getUserId() ) ) {
							continue 2;
						}

						$notificationsStream		=	new Notifications( $asset, $actionOwner );

						$notificationsStream->set( 'parent', $this->notificationString( $userId, $actionOwner, $row->getString( 'parent', '' ) ) );
						$notificationsStream->set( 'moderators', explode( ',', $this->notificationString( $userId, $actionOwner, $row->getString( 'moderators', '' ) ) ) );
						$notificationsStream->set( 'autoaction', $this->autoaction()->getId() );

						$notificationsParams		=	$row->subTree( 'notifications_stream' );

						$notificationsStream->parse( $notificationsParams, 'notifications_' );

						$layout						=	$notificationsStream->getString( 'layout', 'button' );

						switch ( $notificationsParams->getString( 'notifications_state', 'unread' ) ) {
							case 'read':
								if ( \in_array( $layout, [ 'button', 'toggle' ], true ) ) {
									$notificationsStream->set( 'read', 'read' );
								} else {
									$notificationsStream->set( 'read', 'readonly' );
								}
								break;
							case 'unread':
								if ( \in_array( $layout, [ 'button', 'toggle' ], true ) ) {
									$notificationsStream->set( 'read', 'unread' );
								} else {
									$notificationsStream->set( 'read', 'unreadonly' );
								}
								break;
							case 'all':
								$notificationsStream->set( 'read', 'status' );
								break;
						}

						if ( \in_array( $layout, [ 'button', 'toggle' ], true ) ) {
							$stream					=	$notificationsStream->notifications( 'button' );
						} else {
							if ( ! $notificationsStream->rows() ) {
								continue 2;
							}

							$stream					=	$notificationsStream->notifications();
						}
						break;
					case 'activity':
					default:
						$activityStream				=	new Activity( $asset, $actionOwner );

						$activityStream->set( 'parent', $this->notificationString( $userId, $actionOwner, $row->getString( 'parent', '' ) ) );
						$activityStream->set( 'moderators', explode( ',', $this->notificationString( $userId, $actionOwner, $row->getString( 'moderators', '' ) ) ) );
						$activityStream->set( 'autoaction', $this->autoaction()->getId() );

						$activityStream->parse( $row->subTree( 'activity_stream' ), 'activity_' );

						if ( ( ! CBActivity::canCreate( 'activity', $activityStream ) ) && ( ! $activityStream->rows() ) ) {
							continue 2;
						}

						if ( \in_array( $activityStream->getString( 'layout', 'stream' ), [ 'button', 'toggle' ], true ) ) {
							$stream					=	$activityStream->activity( 'button' );
						} else {
							$stream					=	$activityStream->activity();
						}
						break;
				}

				if ( ! $stream ) {
					continue;
				}

				$return								.=	$stream;
			} elseif ( $method === 'delete' ) {
				switch ( $mode ) {
					case 'comment':
						$table						=	'#__comprofiler_plugin_activity_comments';
						$class						=	'\CB\Plugin\Activity\Table\CommentTable';
						break;
					case 'hidden':
						$table						=	'#__comprofiler_plugin_activity_hidden';
						$class						=	'\CB\Plugin\Activity\Table\HiddenTable';
						break;
					case 'follow':
						$table						=	'#__comprofiler_plugin_activity_following';
						$class						=	'\CB\Plugin\Activity\Table\FollowTable';
						break;
					case 'like':
						$table						=	'#__comprofiler_plugin_activity_likes';
						$class						=	'\CB\Plugin\Activity\Table\LikeTable';
						break;
					case 'tag':
						$table						=	'#__comprofiler_plugin_activity_tags';
						$class						=	'\CB\Plugin\Activity\Table\TagTable';
						break;
					case 'notification':
						$table						=	'#__comprofiler_plugin_activity_notifications';
						$class						=	'\CB\Plugin\Activity\Table\NotificationTable';
						break;
					case 'activity':
					default:
						$table						=	'#__comprofiler_plugin_activity';
						$class						=	'\CB\Plugin\Activity\Table\ActivityTable';

						if ( ! $asset ) {
							$asset					=	'profile.' . $actionOwner->getInt( 'id', 0 );
						}
						break;
				}

				$where								=	[];

				if ( $mode === 'hidden' ) {
					$item							=	$this->notificationString( $userId, $actionOwner, $row->getString( 'item', '' ) );

					if ( ! $item ) {
						$this->error( CBTxt::T( 'AUTO_ACTION_ACTIVITY_NO_ITEM', ':: Action [action] :: CB Activity skipped due to missing item', [ '[action]' => $this->autoaction()->getId() ] ) );
						continue;
					}

					$where[]						=	$_CB_database->NameQuote( 'user_id' ) . ' = ' . $actionOwner->getInt( 'id', 0 );
					$where[]						=	$_CB_database->NameQuote( 'type' ) . ' = ' . $_CB_database->Quote( $row->getString( 'type', 'activity' ) );

					switch ( $row->getString( 'type', 'activity' ) ) {
						case 'activity.asset':
						case 'comment.asset':
						case 'notification.asset':
							$where[]				=	$_CB_database->NameQuote( 'asset' ) . ( strpos( $item, '%' ) !== false ? ' LIKE ' : ' = ' ) . $_CB_database->Quote( $item );
							break;
						case 'activity':
						case 'activity.user':
						case 'comment':
						case 'comment.user':
						case 'notification':
						case 'notification.user':
						default:
							$where[]				=	$_CB_database->NameQuote( 'object' ) . ( strpos( $item, '%' ) !== false ? ' LIKE ' : ' = ' ) . $_CB_database->Quote( $item );
							break;
					}
				} else {
					$assetsWhere					=	$this->getAssetWhere( $asset );

					switch ( $row->getString( 'delete_by', 'asset' ) ) {
						case 'owner':
							$where[]				=	$_CB_database->NameQuote( 'user_id' ) . ' = ' . $actionOwner->getInt( 'id', 0 );
							break;
						case 'user':
							if ( $mode !== 'notification' ) {
								$this->error( CBTxt::T( 'AUTO_ACTION_ACTIVITY_WRONG_MODE', ':: Action [action] :: CB Activity skipped due to wrong mode', [ '[action]' => $this->autoaction()->getId() ] ) );
								continue 2;
							}

							$where[]				=	$_CB_database->NameQuote( 'user' ) . ' = ' . $actionUser->getInt( 'id', 0 );
							break;
						case 'owner_user':
							if ( $mode !== 'notification' ) {
								$this->error( CBTxt::T( 'AUTO_ACTION_ACTIVITY_WRONG_MODE', ':: Action [action] :: CB Activity skipped due to wrong mode', [ '[action]' => $this->autoaction()->getId() ] ) );
								continue 2;
							}

							$where[]				=	$_CB_database->NameQuote( 'user_id' ) . ' = ' . $actionOwner->getInt( 'id', 0 );
							$where[]				=	$_CB_database->NameQuote( 'user' ) . ' = ' . $actionUser->getInt( 'id', 0 );
							break;
						case 'asset_user':
							if ( ! $asset ) {
								$this->error( CBTxt::T( 'AUTO_ACTION_ACTIVITY_NO_ASSET', ':: Action [action] :: CB Activity skipped due to missing asset', [ '[action]' => $this->autoaction()->getId() ] ) );
								continue 2;
							}

							if ( $mode !== 'notification' ) {
								$this->error( CBTxt::T( 'AUTO_ACTION_ACTIVITY_WRONG_MODE', ':: Action [action] :: CB Activity skipped due to wrong mode', [ '[action]' => $this->autoaction()->getId() ] ) );
								continue 2;
							}

							$where[]				=	$_CB_database->NameQuote( 'user' ) . ' = ' . $actionUser->getInt( 'id', 0 );

							if ( $assetsWhere ) {
								$where[]			=	$assetsWhere;
							}
							break;
						case 'asset_owner_user':
							if ( ! $asset ) {
								$this->error( CBTxt::T( 'AUTO_ACTION_ACTIVITY_NO_ASSET', ':: Action [action] :: CB Activity skipped due to missing asset', [ '[action]' => $this->autoaction()->getId() ] ) );
								continue 2;
							}

							if ( $mode !== 'notification' ) {
								$this->error( CBTxt::T( 'AUTO_ACTION_ACTIVITY_WRONG_MODE', ':: Action [action] :: CB Activity skipped due to wrong mode', [ '[action]' => $this->autoaction()->getId() ] ) );
								continue 2;
							}

							$where[]				=	$_CB_database->NameQuote( 'user_id' ) . ' = ' . $actionOwner->getInt( 'id', 0 );
							$where[]				=	$_CB_database->NameQuote( 'user' ) . ' = ' . $actionUser->getInt( 'id', 0 );

							if ( $assetsWhere ) {
								$where[]			=	$assetsWhere;
							}
							break;
						case 'asset_owner':
							if ( ! $asset ) {
								$this->error( CBTxt::T( 'AUTO_ACTION_ACTIVITY_NO_ASSET', ':: Action [action] :: CB Activity skipped due to missing asset', [ '[action]' => $this->autoaction()->getId() ] ) );
								continue 2;
							}

							$where[]				=	$_CB_database->NameQuote( 'user_id' ) . ' = ' . $actionOwner->getInt( 'id', 0 );

							if ( $assetsWhere ) {
								$where[]			=	$assetsWhere;
							}
							break;
						case 'asset':
						default:
							if ( ! $asset ) {
								$this->error( CBTxt::T( 'AUTO_ACTION_ACTIVITY_NO_ASSET', ':: Action [action] :: CB Activity skipped due to missing asset', [ '[action]' => $this->autoaction()->getId() ] ) );
								continue 2;
							}

							if ( $assetsWhere ) {
								$where[]			=	$assetsWhere;
							}
							break;
					}
				}

				$query								=	'SELECT *'
													.	"\n FROM " . $_CB_database->NameQuote( $table )
													.	"\n WHERE " . implode( "\n AND ", $where );
				$_CB_database->setQuery( $query );
				$objects							=	$_CB_database->loadObjectList( null, $class, [ $_CB_database ] );

				/** @var ActivityTable[]|NotificationTable[]|CommentTable[]|HiddenTable[]|FollowTable[]|LikeTable[]|TagTable[] $objects */
				foreach ( $objects as $object ) {
					$object->delete();
				}
			} else {
				if ( ! $actionOwner->getInt( 'id', 0 ) ) {
					$this->error( CBTxt::T( 'AUTO_ACTION_ACTIVITY_NO_OWNER', ':: Action [action] :: CB Activity skipped due to missing owner', [ '[action]' => $this->autoaction()->getId() ] ) );
					continue;
				}

				switch ( $mode ) {
					case 'comment':
						$object						=	new CommentTable();
						break;
					case 'hidden':
						$object						=	new HiddenTable();
						break;
					case 'follow':
						$object						=	new FollowTable();
						break;
					case 'like':
						$object						=	new LikeTable();
						break;
					case 'tag':
						$object						=	new TagTable();
						break;
					case 'notification':
						$object						=	new NotificationTable();
						break;
					case 'activity':
					default:
						$object						=	new ActivityTable();
						break;
				}

				if ( $mode === 'hidden' ) {
					$item							=	$this->notificationString( $userId, $actionOwner, $row->getString( 'item', '' ) );

					if ( ! $item ) {
						$this->error( CBTxt::T( 'AUTO_ACTION_ACTIVITY_NO_ITEM', ':: Action [action] :: CB Activity skipped due to missing item', [ '[action]' => $this->autoaction()->getId() ] ) );
						continue;
					}

					switch ( $row->getString( 'type', 'activity' ) ) {
						case 'activity.asset':
						case 'comment.asset':
						case 'notification.asset':
							$object->load( [ 'user_id' => $actionOwner->getInt( 'id', 0 ), 'type' => $row->getString( 'type', 'activity' ), 'asset' => $item ] );

							$object->set( 'asset', $item );
							break;
						case 'activity':
						case 'activity.user':
						case 'comment':
						case 'comment.user':
						case 'notification':
						case 'notification.user':
						default:
							$object->load( [ 'user_id' => $actionOwner->getInt( 'id', 0 ), 'type' => $row->getString( 'type', 'activity' ), 'object' => $item ] );

							$object->set( 'object', $item );
							break;
					}
				} else {
					if ( ! $asset ) {
						$this->error( CBTxt::T( 'AUTO_ACTION_ACTIVITY_NO_ASSET', ':: Action [action] :: CB Activity skipped due to missing asset', [ '[action]' => $this->autoaction()->getId() ] ) );
						continue;
					}

					switch ( $row->getString( 'create_by', 'asset' ) ) {
						case 'owner':
							$object->load( [ 'user_id' => $actionOwner->getInt( 'id', 0 ) ] );
							break;
						case 'user':
							if ( $mode !== 'notification' ) {
								$this->error( CBTxt::T( 'AUTO_ACTION_ACTIVITY_WRONG_MODE', ':: Action [action] :: CB Activity skipped due to wrong mode', [ '[action]' => $this->autoaction()->getId() ] ) );
								continue 2;
							}

							$object->load( [ 'user' => $actionUser->getInt( 'id', 0 ) ] );
							break;
						case 'owner_user':
							if ( $mode !== 'notification' ) {
								$this->error( CBTxt::T( 'AUTO_ACTION_ACTIVITY_WRONG_MODE', ':: Action [action] :: CB Activity skipped due to wrong mode', [ '[action]' => $this->autoaction()->getId() ] ) );
								continue 2;
							}

							$object->load( [ 'user_id' => $actionOwner->getInt( 'id', 0 ), 'user' => $actionUser->getInt( 'id', 0 ) ] );
							break;
						case 'asset_user':
							if ( $mode !== 'notification' ) {
								$this->error( CBTxt::T( 'AUTO_ACTION_ACTIVITY_WRONG_MODE', ':: Action [action] :: CB Activity skipped due to wrong mode', [ '[action]' => $this->autoaction()->getId() ] ) );
								continue 2;
							}

							$object->load( [ 'user' => $actionUser->getInt( 'id', 0 ), 'asset' => $asset ] );
							break;
						case 'asset_owner_user':
							if ( $mode !== 'notification' ) {
								$this->error( CBTxt::T( 'AUTO_ACTION_ACTIVITY_WRONG_MODE', ':: Action [action] :: CB Activity skipped due to wrong mode', [ '[action]' => $this->autoaction()->getId() ] ) );
								continue 2;
							}

							$object->load( [ 'user_id' => $actionOwner->getInt( 'id', 0 ), 'user' => $actionUser->getInt( 'id', 0 ), 'asset' => $asset ] );
							break;
						case 'asset_owner':
							$object->load( [ 'user_id' => $actionOwner->getInt( 'id', 0 ), 'asset' => $asset ] );
							break;
						case 'asset':
							$object->load( [ 'asset' => $asset ] );
							break;
					}

					$date							=	$this->notificationString( $userId, $actionOwner, $row->getString( 'date', '' ) );

					if ( $date ) {
						try {
							$date					=	Application::Date( $date, 'UTC' )->format( 'Y-m-d H:i:s' );
						} catch ( \Exception $e ) {
							$date					=	'';
						}
					}

					$group							=	$this->notificationString( $userId, $actionOwner, $row->getString( 'group', '' ) );

					if ( \in_array( $mode, [ 'activity', 'notification' ], true ) && $group && ( ! $object->getInt( 'id', 0 ) ) ) {
						switch ( $mode ) {
							case 'notification':
								$table				=	'#__comprofiler_plugin_activity_notifications';
								$class				=	'\CB\Plugin\Activity\Table\NotificationTable';
								break;
							case 'activity':
							default:
								$table				=	'#__comprofiler_plugin_activity';
								$class				=	'\CB\Plugin\Activity\Table\ActivityTable';
								break;
						}

						$where						=	[];

						switch ( $row->getString( 'group_by', 'group' ) ) {
							case 'group_user':
								if ( $mode !== 'notification' ) {
									$this->error( CBTxt::T( 'AUTO_ACTION_ACTIVITY_WRONG_MODE', ':: Action [action] :: CB Activity skipped due to wrong mode', [ '[action]' => $this->autoaction()->getId() ] ) );
									continue 2;
								}

								$where[]			=	$_CB_database->NameQuote( 'user' ) . ' = ' . $actionUser->getInt( 'id', 0 );
								break;
							case 'group_owner_user':
								if ( $mode !== 'notification' ) {
									$this->error( CBTxt::T( 'AUTO_ACTION_ACTIVITY_WRONG_MODE', ':: Action [action] :: CB Activity skipped due to wrong mode', [ '[action]' => $this->autoaction()->getId() ] ) );
									continue 2;
								}

								$where[]			=	$_CB_database->NameQuote( 'user_id' ) . ' = ' . $actionOwner->getInt( 'id', 0 );
								$where[]			=	$_CB_database->NameQuote( 'user' ) . ' = ' . $actionUser->getInt( 'id', 0 );
								break;
							case 'group_owner':
								$where[]			=	$_CB_database->NameQuote( 'user_id' ) . ' = ' . $actionOwner->getInt( 'id', 0 );
								break;
						}

						$where[]					=	$this->getAssetWhere( $group );

						$groupOffset				=	$this->notificationString( $userId, $actionOwner, $row->getString( 'group_offset', '15 MINUTES' ) );

						if ( $groupOffset ) {
							$groupOffset			=	str_replace( [ '-', '+' ], '', $groupOffset );

							if ( is_numeric( $groupOffset ) ) {
								$groupOffset		.=	' MINUTES';
							}

							try {
								$groupDate			=	Application::Date( ( $date ?: 'now' ), 'UTC' );

								$where[]			=	$_CB_database->NameQuote( 'date' ) . " BETWEEN " . $_CB_database->Quote( $groupDate->modify( '-' . strtoupper( $groupOffset ) )->format( 'Y-m-d H:i:s' ) ) . " AND " . $_CB_database->Quote( $groupDate->modify( '+' . strtoupper( $groupOffset ) )->format( 'Y-m-d H:i:s' ) );
							} catch ( \Exception $e ) {}
						}

						$query						=	'SELECT *'
													.	"\n FROM " . $_CB_database->NameQuote( $table )
													.	"\n WHERE " . implode( "\n AND ", $where )
													.	"\n ORDER BY " . $_CB_database->NameQuote( 'date' ) . " DESC";
						$_CB_database->setQuery( $query, 0, 1 );
						$groupObject				=	new $class();
						$_CB_database->loadObject( $groupObject );

						/** @var ActivityTable|NotificationTable $groupObject */
						if ( $groupObject->getInt( 'id', 0 ) ) {
							$groupedAssets			=	$groupObject->params()->getRaw( 'assets', [] );

							$groupedAssets[]		=	$asset;

							$groupObject->params()->set( 'assets', $groupedAssets );

							$groupObject->set( 'params', $groupObject->params()->asJson() );

							if ( ! $groupObject->store() ) {
								$this->error( CBTxt::T( 'AUTO_ACTION_ACTIVITY_CREATE_FAILED', ':: Action [action] :: CB Activity failed to save. Error: [error]', [ '[action]' => $this->autoaction()->getId(), '[error]' => $groupObject->getError() ] ) );
							}

							continue;
						}
					}

					$object->set( 'asset', $asset );

					if ( \in_array( $mode, [ 'activity', 'notification' ], true ) ) {
						$title						=	$this->notificationString( $userId, $actionOwner, $row->getRaw( 'title', '' ), false );

						if ( $title ) {
							$object->set( 'title', $title );
						}
					} elseif ( $mode === 'like' ) {
						$object->set( 'type', $row->getInt( 'type', 0 ) );
					} elseif ( $mode === 'tag' ) {
						$tag						=	$this->notificationString( $userId, $actionOwner, $row->getString( 'tag', '' ) );

						if ( ! $tag ) {
							$this->error( CBTxt::T( 'AUTO_ACTION_ACTIVITY_NO_TAG', ':: Action [action] :: CB Activity skipped due to missing tag', [ '[action]' => $this->autoaction()->getId() ] ) );
							continue;
						}

						$object->set( 'tag', $tag );
					}

					if ( $mode === 'notification' ) {
						$link						=	$this->notificationString( $userId, $actionOwner, $row->getString( 'link', '' ), false );

						if ( $link ) {
							$object->set( 'link', $link );
						}
					}

					if ( \in_array( $mode, [ 'activity', 'notification', 'comment' ], true ) ) {
						$message					=	$this->notificationString( $userId, $actionOwner, $row->getString( 'message', '' ), false );

						if ( $message ) {
							// Remove duplicate spaces:
							$message				=	preg_replace( '/ {2,}/', ' ', $message );
							// Remove duplicate tabs:
							$message				=	preg_replace( '/\t{2,}/', "\t", $message );
							// Remove duplicate linebreaks:
							$message				=	preg_replace( '/(\r\n|\r|\n){2,}/i', '$1', $message );

							$object->set( 'message', $message );
						}

						if ( \in_array( $mode, [ 'activity', 'comment' ], true ) ) {
							$action					=	$row->subTree( 'action' );
							$actionId				=	$action->getInt( 'id', 0 );

							if ( $actionId ) {
								$actionMessage		=	$this->notificationString( $userId, $actionOwner, $action->getString( 'message', '' ), false );

								if ( $actionMessage ) {
									// Remove linebreaks:
									$actionMessage	=	str_replace( [ "\n", "\r\n" ], ' ', $actionMessage );
									// Remove duplicate spaces:
									$actionMessage	=	preg_replace( '/ {2,}/', ' ', $actionMessage );
									// Remove duplicate tabs:
									$actionMessage	=	preg_replace( '/\t{2,}/i', "\t", $actionMessage );

									$newAction		=	[	'id'		=>	$actionId,
															'message'	=>	$actionMessage,
															'emote'		=>	$action->getInt( 'emote', 0 )
														];

									$object->params()->set( 'action', $newAction );
								}
							}

							$location				=	$row->subTree( 'location' );
							$locationId				=	$location->getInt( 'id', 0 );

							if ( $locationId ) {
								$locationPlace		=	$this->notificationString( $userId, $actionOwner, $location->getString( 'place', '' ), false );

								if ( $locationPlace ) {
									$newLocation	=	[	'id'		=>	$locationId,
															'place'		=>	$locationPlace,
															'address'	=>	$this->notificationString( $userId, $actionOwner, $location->getString( 'address', '' ), false )
														];

									$object->params()->set( 'location', $newLocation );
								}
							}
						}

						$newLinks					=	[];

						foreach ( $row->subTree( 'links' ) as $link ) {
							/** @var ParamsInterface $link */
							$linkType				=	$link->getString( 'type', '' );
							$linkUrl				=	$this->notificationString( $userId, $actionOwner, $link->getString( 'url', '' ), false );

							if ( ( ! $linkType ) || ( ( $linkType !== 'custom' ) && ( ! $linkUrl ) ) ) {
								continue;
							}

							$linkMedia				=	$link->subTree( 'media' );
							$linkMediaURL			=	$this->notificationString( $userId, $actionOwner, $linkMedia->getString( 'url', '' ), false );
							$linkMediaCustom		=	$this->notificationString( $userId, $actionOwner, $linkMedia->getRaw( 'custom', '' ), false );

							if ( ( $linkType === 'custom' ) && ( ! $linkMediaCustom ) ) {
								continue;
							}

							$linkDate				=	$this->notificationString( $userId, $actionOwner, $link->getString( 'date', '' ) );

							if ( $linkDate ) {
								try {
									$linkDate		=	Application::Date( $linkDate, 'UTC' )->getTimestamp();
								} catch ( \Exception $e ) {
									$linkDate		=	'';
								}
							} else {
								$linkDate			=	Application::Database()->getUtcDateTime();
							}

							$newLinks[]				=	[	'type'			=>	$linkType,
															'url'			=>	trim( $linkUrl ),
															'title'			=>	trim( $this->notificationString( $userId, $actionOwner, $link->getString( 'title', '' ), false ) ),
															'description'	=>	trim( $this->notificationString( $userId, $actionOwner, $link->getString( 'description', '' ), false ) ),
															'media'			=>	[	'type'		=>	$linkType,
																					'url'		=>	$linkMediaURL,
																					'filename'	=>	$this->notificationString( $userId, $actionOwner, $linkMedia->getString( 'filename', '' ) ),
																					'mimetype'	=>	$this->notificationString( $userId, $actionOwner, $linkMedia->getString( 'mimetype', '' ) ),
																					'extension'	=>	$this->notificationString( $userId, $actionOwner, $linkMedia->getString( 'extension', '' ) ),
																					'filesize'	=>	(int) $this->notificationString( $userId, $actionOwner, $linkMedia->getString( 'filesize', '' ) ),
																					'custom'	=>	$linkMediaCustom,
																					'internal'	=>	\JUri::isInternal( $linkMediaURL )
																				],
															'thumbnails'	=>	[],
															'selected'		=>	0,
															'thumbnail'		=>	$link->getBool( 'thumbnail', true ),
															'internal'		=>	\JUri::isInternal( $linkUrl ),
															'date'			=>	$linkDate,
															'embedded'		=>	false
														];
						}

						if ( $newLinks ) {
							$object->params()->set( 'links', $newLinks );
						}

						if ( $mode === 'activity' ) {
							$row->set( 'defaults.tags_asset', $this->notificationString( $userId, $actionOwner, $row->getString( 'defaults.tags_asset', '' ) ) );
							$row->set( 'defaults.tags_user', $this->notificationString( $userId, $actionOwner, $row->getString( 'defaults.tags_user', '' ) ) );

							$row->set( 'defaults.likes_asset', $this->notificationString( $userId, $actionOwner, $row->getString( 'defaults.likes_asset', '' ) ) );
							$row->set( 'defaults.likes_user', $this->notificationString( $userId, $actionOwner, $row->getString( 'defaults.likes_user', '' ) ) );

							$row->set( 'defaults.comments_asset', $this->notificationString( $userId, $actionOwner, $row->getString( 'defaults.comments_asset', '' ) ) );
							$row->set( 'defaults.comments_user', $this->notificationString( $userId, $actionOwner, $row->getString( 'defaults.comments_user', '' ) ) );

							$object->params()->set( 'defaults', $row->subTree( 'defaults' )->asArray() );
						}

						$object->set( 'params', $object->params()->asJson() );

						$published					=	$row->getString( 'published', '' );

						if ( ( $published === null ) || ( $published === '' ) ) {
							$object->set( 'published', 1 );
						} else {
							$object->set( 'published', ( (int) $this->notificationString( $userId, $actionOwner, $published ) >= 1 ? 1 : 0 ) );
						}

						if ( \in_array( $mode, [ 'activity', 'notification', 'comment' ], true ) ) {
							$system					=	$row->getString( 'system', '' );

							if ( ( $system === null ) || ( $system === '' ) ) {
								$object->set( 'system', 0 );
							} else {
								$object->set( 'system', ( (int) $this->notificationString( $userId, $actionOwner, $system ) ? 1 : 0 ) );
							}
						}

						if ( \in_array( $mode, [ 'activity', 'notification' ], true ) ) {
							$global					=	$row->getString( 'global', '' );

							if ( ( $global === null ) || ( $global === '' ) ) {
								$object->set( 'global', 0 );
							} else {
								$object->set( 'global', ( (int) $this->notificationString( $userId, $actionOwner, $global ) ? 1 : 0 ) );
							}
						}

						$pinned						=	$row->getString( 'pinned', '' );

						if ( ( $pinned === null ) || ( $pinned === '' ) ) {
							$object->set( 'pinned', 0 );
						} else {
							$object->set( 'pinned', ( (int) $this->notificationString( $userId, $actionOwner, $pinned ) ? 1 : 0 ) );
						}
					}

					if ( $date ) {
						$object->set( 'date', $date );
					}
				}

				$object->set( 'user_id', $actionOwner->getInt( 'id', 0 ) );

				if ( $mode === 'notification' ) {
					$object->set( 'user', $actionUser->getInt( 'id', 0 ) );
				}

				if ( ! $object->store() ) {
					$this->error( CBTxt::T( 'AUTO_ACTION_ACTIVITY_CREATE_FAILED', ':: Action [action] :: CB Activity failed to save. Error: [error]', [ '[action]' => $this->autoaction()->getId(), '[error]' => $object->getError() ] ) );
				}
			}
		}

		return $return;
	}

	/**
	 * Parses a string through action substitutions
	 *
	 * @param int             $recipient
	 * @param null|UserTable  $user
	 * @param null|string     $string
	 * @param null|array|bool $htmlspecialchars
	 * @param null|bool       $translate
	 * @param null|bool       $format
	 * @param null|bool       $prepare
	 * @param null|bool       $substitutions
	 * @param array           $extras
	 * @return string
	 */
	private function notificationString( int $recipient, ?UserTable $user, ?string $string, $htmlspecialchars = true, ?bool $translate = null, ?bool $format = null, ?bool $prepare = null, ?bool $substitutions = null, array $extras = [] ): string
	{
		$string					=	str_ireplace( '[recipient]', $recipient, $string );

		$extras['recipient']	=	$recipient;

		return $this->string( $user, $string, $htmlspecialchars, $translate, $format, $prepare, $substitutions, $extras );
	}

	/**
	 * Returns the database where statement from asset
	 *
	 * @param null|string $asset
	 * @return string
	 */
	private function getAssetWhere( ?string $asset ): string
	{
		global $_CB_database;

		if ( ! $asset ) {
			return '';
		}

		$where						=	[];
		$queryAssets				=	[];
		$queryWildcards				=	[];

		foreach ( explode( ',', $asset ) as $queryAsset ) {
			if ( ( strpos( $queryAsset, '%' ) !== false ) || ( strpos( $queryAsset, '_' ) !== false ) ) {
				$queryWildcards[]	=	$queryAsset;
			} else {
				$queryAssets[]		=	$queryAsset;
			}
		}

		$queryAssets				=	array_unique( $queryAssets );
		$queryWildcards				=	array_unique( $queryWildcards );

		if ( $queryAssets ) {
			$where[]				=	$_CB_database->NameQuote( 'asset' ) . ( \count( $queryAssets ) > 1 ? " IN " . $_CB_database->safeArrayOfStrings( $queryAssets ) : " = " . $_CB_database->Quote( $queryAssets[0] ) );
		}

		foreach ( $queryWildcards as $queryWildcard ) {
			$where[]				=	$_CB_database->NameQuote( 'asset' ) . " LIKE " . $_CB_database->Quote( $queryWildcard );
		}

		return ( \count( $where ) > 1 ? "( " . implode( " OR ", $where ) . " )" : $where[0] );
	}

	/**
	 * @return bool
	 */
	public function installed(): bool
	{
		global $_PLUGINS;

		$plugin				=	$_PLUGINS->getLoadedPlugin( 'user', 'cbactivity' );

		if ( ! $plugin ) {
			return false;
		}

		$pluginVersion		=	str_replace( '+build.', '+', $_PLUGINS->getPluginVersion( $plugin, true ) );

		return version_compare( $pluginVersion, '5.0.0', '>=' ) && version_compare( $pluginVersion, '6.0.0', '<' );
	}
}