<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\Privacy\Trigger;

use CBLib\Application\Application;
use CBLib\Language\CBTxt;
use CBLib\Registry\GetterInterface;
use CB\Database\Table\UserTable;
use CB\Plugin\Privacy\CBPrivacy;
use CB\Plugin\Privacy\Privacy;
use CB\Database\Table\TabTable;
use CBLib\Registry\ParamsInterface;
use CBLib\Registry\Registry;

defined('CBLIB') or die();

class TabTrigger extends \cbPluginHandler
{

	/**
	 * @param null|TabTable $tab
	 * @param UserTable       $user
	 * @param string          $reason
	 * @return string
	 */
	private function tabOverride( $tab, $user, $reason )
	{
		if ( ! $tab->params instanceof ParamsInterface ) {
			$tab->params	=	new Registry( $tab->params );
		}

		switch( $reason ) {
			case 'register':
			case 'edit':
				$override	=	CBTxt::T( $tab->params->get( 'cbprivacy_edit_override', null, GetterInterface::RAW ) );
				break;
			case 'list':
			case 'profile':
			default:
				$override	=	CBTxt::T( $tab->params->get( 'cbprivacy_display_override', null, GetterInterface::RAW ) );
				break;
		}

		$override			=	trim( preg_replace( '/\[cb:usertab tab="' . preg_quote( $tab->getInt( 'tabid', 0 ), '/' ) . '"[^]]+\]/i', '', $override ) );

		if ( $override ) {
			$override		=	\CBuser::getInstance( $user->get( 'id', 0, GetterInterface::INT ) )->replaceUserVars( $override, false, false, array(), false );
		}

		return $override;
	}

	/**
	 * @param array     $tabsContents
	 * @param array     $tabsToDisplay
	 * @param UserTable $user
	 * @param string    $position
	 * @param int       $tabId
	 */
	public function tabsDisplay( &$tabsContents, &$tabsToDisplay, &$user, $position, $tabId )
	{
		foreach ( $tabsToDisplay as $tabId => $tab ) {
			if ( ! CBPrivacy::checkTabDisplayAccess( $tab, $user ) ) {
				$content				=	$this->tabOverride( $tab, $user, 'profile' );

				if ( ! $content ) {
					$content			=	null;
				}

				$tabsContents[$tabId]	=	$content;
			}
		}
	}

	/**
	 * @param string    $content
	 * @param TabTable  $tab
	 * @param UserTable $user
	 * @param array     $postdata
	 * @param string    $output
	 * @param string    $formatting
	 * @param string    $reason
	 * @param bool      $tabbed
	 */
	public function tabEdit( &$content, &$tab, &$user, &$postdata, $output, $formatting, $reason, $tabbed )
	{
		if ( ( ! in_array( $reason, array( 'edit', 'register' ) ) ) || ( ! $tab instanceof TabTable ) ) {
			return;
		}

		$tabId								=	$tab->get( 'tabid', 0, GetterInterface::INT );

		if ( ( $reason != 'register' ) && ( ! CBPrivacy::checkTabEditAccess( $tab ) ) ) {
			$content						=	$this->tabOverride( $tab, $user, $reason );

			if ( ! $content ) {
				$content					=	' ';
			}
		} else {
			if ( ! $tab->params instanceof ParamsInterface ) {
				$tab->params				=	new Registry( $tab->params );
			}

			$display						=	$tab->params->get( 'cbprivacy_display', 0, GetterInterface::INT );

			if ( ( $reason == 'register' ) && ( ! $tab->params->get( 'cbprivacy_display_reg', false, GetterInterface::BOOLEAN ) ) ) {
				$display					=	0;
			}

			if ( ( $display == 1 ) || ( ( $display == 2 ) && Application::MyUser()->isGlobalModerator() ) ) {
				$asset						=	str_replace( '[tab_id]', $tabId, $tab->params->getString( 'cbprivacy_asset', '' ) );

				if ( ! $asset ) {
					$asset					=	'profile.tab.' . $tabId;
				}

				$privacy					=	new Privacy( $asset, $user );

				if ( $display == 2 ) {
					$privacy->set( 'options_moderator', true );
				}

				$privacy->parse( $tab->params, 'privacy_' );

				if ( in_array( $reason, array( 'edit', 'register' ) ) ) {
					$privacy->set( 'ajax', false );
				}

				$privacy->set( 'tab', $tabId );

				if ( ( ( ( $reason == 'edit' ) && CBPrivacy::getGlobalParams()->get( 'profile_simple_edit', false, GetterInterface::BOOLEAN ) )
					   || ( ( $reason == 'register' ) && CBPrivacy::getGlobalParams()->get( 'profile_simple_registration', false, GetterInterface::BOOLEAN ) )
					 ) && ( ! in_array( $tab->get( 'tabid', 0, GetterInterface::INT ), explode( '|*|', CBPrivacy::getGlobalParams()->getString( 'profile_simple_exclude_tabs', '' ) ) ) )
				) {
					CBPrivacy::getGroupedPrivacy( $tab, $privacy );
				} else {
					$selector				=	$privacy->privacy( 'edit' );

					if ( $selector ) {
						switch ( $formatting ) {
							case 'tabletrs':
								$return		=	'<tr id="cbtp_' . (int) $tabId . '" class="cb_table_line cbft_privacy cbtt_select cb_table_line_field">'
											.		'<td class="fieldCell text-right" colspan="2" style="width: 100%;">'
											.			$selector
											.		'</td>'
											.	'</tr>';
								break;
							default:
								$return		=	'<div class="cbft_privacy cbtt_select form-group cb_form_line clearfix cbtwolinesfield" id="cbtp_' . (int) $tabId . '">'
											.		'<div class="cb_field col-sm-12">'
											.			'<div class="text-right">'
											.				$selector
											.			'</div>'
											.		'</div>'
											.	'</div>';
								break;
						}

						$content			=	$return
											.	$content;
					}
				}
			}
		}
	}

	/**
	 * @param TabTable[] $tabs
	 * @param UserTable  $user
	 * @param string     $reason
	 */
	public function tabsFetch( &$tabs, &$user, $reason )
	{
		if ( ( ! $user instanceof UserTable ) || $user->getError() || ( ! $tabs ) ) {
			return;
		}

		if ( ( $reason == 'profile' ) && ( ! Application::Cms()->getClientId() ) && ( ! Application::MyUser()->isGlobalModerator() ) ) {
			foreach ( $tabs as $tabId => $tab ) {
				if ( isset( $tabs[$tabId] ) && ( ! CBPrivacy::checkTabDisplayAccess( $tab, $user ) ) && ( ! $this->tabOverride( $tab, $user, $reason ) ) ) {
					unset( $tabs[$tabId] );
				}
			}
		} elseif ( in_array( $reason, array( 'edit', 'editsave' ) ) && $user->get( 'id', 0, GetterInterface::INT ) ) {
			foreach ( $tabs as $tabId => $tab ) {
				if ( isset( $tabs[$tabId] ) && ( ! CBPrivacy::checkTabEditAccess( $tab ) ) && ( ! $this->tabOverride( $tab, $user, $reason ) ) ) {
					unset( $tabs[$tabId] );
				}
			}
		}

		// Add the generated tab if we're using simplified privacy controls:
		if ( ( ( $reason == 'edit' ) && CBPrivacy::getGlobalParams()->get( 'profile_simple_edit', false, GetterInterface::BOOLEAN ) )
			 || ( ( $reason == 'register' ) && CBPrivacy::getGlobalParams()->get( 'profile_simple_registration', false, GetterInterface::BOOLEAN ) )
		) {
			$simpleTab		=	new TabTable();

			$simpleTab->set( 'title', CBPrivacy::getGlobalParams()->get( 'profile_simple_title', 'Privacy', GetterInterface::HTML ) );
			$simpleTab->set( 'description', CBPrivacy::getGlobalParams()->get( 'profile_simple_description', null, GetterInterface::HTML ) );
			$simpleTab->set( 'ordering', 999 );
			$simpleTab->set( 'ordering_edit', 999 );
			$simpleTab->set( 'ordering_register', 999 );
			$simpleTab->set( 'enabled', 1 );
			$simpleTab->set( 'pluginclass', 'cbprivacyTab' );
			$simpleTab->set( 'pluginid', $this->getPluginId() );
			$simpleTab->set( 'fields', 0 );
			$simpleTab->set( 'sys', 1 );
			$simpleTab->set( 'displaytype', 'html' );
			$simpleTab->set( 'position', 'not_on_profile_1' );
			$simpleTab->set( 'viewaccesslevel', 1 );
			$simpleTab->set( 'cssclass', 'cbPrivacySimpleTab' );
			$simpleTab->set( 'params', new Registry() );

			$tabs[]			=	$simpleTab;
		}
	}
}