<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\GroupJive\Trigger;

use CBLib\Language\CBTxt;
use CBLib\Application\Application;
use CB\Plugin\GroupJive\CBGroupJive;

defined('CBLIB') or die();

class RouterTrigger extends \cbPluginHandler
{

	/**
	 * @param \ComprofilerRouter $router
	 * @param string             $plugin
	 * @param array              $segments
	 * @param array              $query
	 * @param \JMenuSite         $menuItem
	 */
	public function build( $router, $plugin, &$segments, &$query, &$menuItem )
	{
		if ( ( $plugin !== 'cbgroupjive' ) || ( ! $query ) || ( ! isset( $query['action'] ) ) ) {
			return;
		}

		$action						=	$query['action'];

		unset( $query['action'] );

		$category					=	null;
		$group						=	null;
		$id							=	null;

		switch ( $action ) {
			case 'categories':
				if ( isset( $query['id'] ) ) {
					$category		=	CBGroupJive::getCategory( (int) $query['id'] );

					unset( $query['id'] );
				}
				break;
			case 'groups':
				if ( isset( $query['id'] ) ) {
					$group			=	CBGroupJive::getGroup( (int) $query['id'] );

					unset( $query['id'] );
				} elseif ( isset( $query['category'] ) ) {
					$category		=	CBGroupJive::getCategory( (int) $query['category'] );

					unset( $query['category'] );
				}
				break;
			case 'invites':
				if ( isset( $query['group'] ) ) {
					$group			=	CBGroupJive::getGroup( (int) $query['group'] );

					unset( $query['group'] );
				} elseif ( isset( $query['id'] ) ) {
					$id				=	$query['id'];
					$group			=	CBGroupJive::getInvite( (int) $id )->group();

					unset( $query['id'] );
				}
				break;
			case 'notifications':
				if ( isset( $query['id'] ) ) {
					$group			=	CBGroupJive::getGroup( (int) $query['id'] );

					unset( $query['id'] );
				}
				break;
			case 'users':
				if ( isset( $query['id'] ) ) {
					$id				=	$query['id'];
					$group			=	CBGroupJive::getUser( (int) $id )->group();

					unset( $query['id'] );
				}
				break;
		}

		$menuAction					=	( $menuItem && isset( $menuItem->query['action'] ) && ( $menuItem->query['action'] !== '' ) ? $menuItem->query['action'] : null );
		$menuId						=	( $menuItem && isset( $menuItem->query['id'] ) && ( $menuItem->query['id'] !== '' ) ? (int) $menuItem->query['id'] : null );

		$hasCategory				=	( $category && $category->getInt( 'id', 0 ) );
		$hasGroup					=	( $group && $group->getInt( 'id', 0 ) );

		$categoryId					=	0;
		$groupId					=	0;

		if ( $hasCategory || $hasGroup ) {
			if ( $hasCategory ) {
				$categoryId			=	$category->getInt( 'id', 0 );
				$categorySEF		=	true;

				if ( ( $menuAction === 'categories' ) && ( $menuId === $categoryId ) ) {
					// We're already on a category menu item and the id matches so no reason to add it to SEF:
					$categorySEF	=	false;
				}

				if ( $categorySEF ) {
					$segments[]		=	$categoryId . '-' . Application::Router()->stringToAlias( CBTxt::T( $category->getString( 'name', $categoryId ) ) );
				}

				if ( $action !== 'categories' ) {
					$segments[]		=	$action;
				}

				if ( $id ) {
					$segments[]		=	$id;
				}
			} elseif ( $hasGroup ) {
				$categoryId			=	$group->category()->getInt( 'id', 0 );
				$groupId			=	$group->getInt( 'id', 0 );

				$categorySEF		=	true;
				$groupSEF			=	true;

				if ( ( $menuAction === 'categories' ) && ( $menuId === $categoryId ) ) {
					// We're already on a category menu item and the id matches so no reason to add it to SEF:
					$categorySEF	=	false;
				} elseif ( ( $menuAction === 'groups' ) && ( $menuId === $groupId ) ) {
					// We're already on a group menu item and the id matches so no reason to add it to SEF:
					$groupSEF		=	false;
					$categorySEF	=	false;
				}

				if ( $categorySEF ) {
					$segments[]		=	$categoryId . '-' . Application::Router()->stringToAlias( CBTxt::T( $group->category()->getString( 'name', $categoryId ) ) );
				}

				if ( $groupSEF ) {
					$segments[]		=	$groupId . '-' . Application::Router()->stringToAlias( CBTxt::T( $group->getString( 'name', $groupId ) ) );
				}

				if ( $action !== 'groups' ) {
					$segments[]		=	$action;
				}

				if ( $id ) {
					$segments[]		=	$id;
				}
			}

			if ( isset( $query['func'] ) ) {
				$func				=	$query['func'];

				unset( $query['func'] );

				if ( $func !== 'show' ) {
					if ( ( ! $categoryId ) && ( ! $groupId ) && in_array( $action, array( 'categories', 'groups' ), true ) ) {
						// We're trying to reach a category/group page but no ids exist so lets re-add the action:
						$segments[]	=	$action;
					}

					$segments[]		=	$func;
				}
			}
		} else {
			$segments[]				=	$action;

			if ( isset( $query['func'] ) ) {
				$segments[]			=	$query['func'];

				unset( $query['func'] );

				if ( isset( $query['id'] ) ) {
					$segments[]		=	$query['id'];

					unset( $query['id'] );
				}
			}
		}
	}

	/**
	 * @param \ComprofilerRouter $router
	 * @param string             $plugin
	 * @param array              $segments
	 * @param array              $vars
	 * @param \JMenuSite         $menuItem
	 */
	public function parse( $router, $plugin, $segments, &$vars, $menuItem )
	{
		if ( ( $plugin !== 'cbgroupjive' ) || ( ! $segments ) ) {
			return;
		}

		$menuAction									=	( $menuItem && isset( $menuItem->query['action'] ) && ( $menuItem->query['action'] !== '' ) ? $menuItem->query['action'] : null );
		$menuId										=	( $menuItem && isset( $menuItem->query['id'] ) && ( $menuItem->query['id'] !== '' ) ? (int) $menuItem->query['id'] : null );

		$menuCategoryId								=	( $menuAction === 'categories' ? $menuId : 0 );

		if ( $menuCategoryId && isset( $segments[1] ) && ( $segments[1] === 'all' ) ) {
			// Menu item is to a specific category, but we're trying to reach the all categories page:
			$menuCategoryId							=	0;
		}

		$menuGroupId								=	( $menuAction === 'groups' ? $menuId : 0 );

		if ( $menuGroupId && isset( $segments[1] ) && in_array( $segments[1], array( 'all', 'my', 'owned', 'joined', 'invited', 'approval' ), true ) ) {
			// Menu item is to a specific group, but we're trying to reach the my groups, joined groups, etc.. page:
			$menuGroupId							=	0;
		}

		// Category id doesn't exist so assume the first segment is the category or group:
		$categoryId								=	( isset( $segments[0] ) ? preg_replace( '/-/', ':', $segments[0], 1 ) : null );

		if ( strpos( $categoryId, ':' ) !== false ) {
			list( $categoryId, $categoryAlias )	=	explode( ':', $categoryId, 2 );
		} else {
			$categoryAlias						=	null;
		}

		unset( $segments[0] );

		// Now assume the second segment is the group or action:
		$groupId								=	( isset( $segments[1] ) ? preg_replace( '/-/', ':', $segments[1], 1 ) : null );

		if ( strpos( $groupId, ':' ) !== false ) {
			list( $groupId, $groupAlias )		=	explode( ':', $groupId, 2 );
		} else {
			$groupAlias							=	null;
		}

		unset( $segments[1] );

		$segments								=	array_values( $segments );
		$count									=	count( $segments );

		if ( is_numeric( $categoryId ) && is_numeric( $groupId ) ) {
			// We have both a category and group id found so lets treat this as a group endpoint:
			$this->groupPath( $categoryId, $groupId, $segments, $vars );
		} elseif ( is_numeric( $categoryId ) ) {
			// We only have a category or group id so lets try to figure out which:
			$group								=	CBGroupJive::getGroup( (int) $categoryId );

			// Looks like we're trying to reach an uncategorized group so lets make sure the alias also matches if one exists to be sure it's not actually a category:
			if ( $group->getInt( 'id', 0 )
				 && ( ( ! $categoryAlias ) || ( $categoryAlias === Application::Router()->stringToAlias( CBTxt::T( $group->getString( 'name' ) ) ) ) )
			) {
				$vars['action']					=	'groups';

				if ( $groupId ) {
					$vars['func']				=	$groupId;

					if ( $vars['func'] === 'invites' ) {
						$vars['group']			=	(int) $categoryId;
					} elseif ( $vars['func'] !== 'users' ) {
						$vars['id']				=	(int) $categoryId;
					}

					if ( ( $count > 0 ) && in_array( $vars['func'], array( 'notifications', 'users', 'invites' ), true ) ) {
						$vars['action']			=	$vars['func'];

						if ( ( $count > 1 ) && in_array( $vars['func'], array( 'users', 'invites' ), true ) ) {
							$vars['func']		=	$segments[1];
							$vars['id']			=	(int) $segments[0];
						} else {
							$vars['func']		=	$segments[0];
						}
					}
				} else {
					$vars['id']					=	(int) $categoryId;
				}
			} else {
				// No group id supplied so lets treat it as a category endpoint:
				$this->categoryPath( $categoryId, $groupId, $segments, $vars );
			}
		} elseif ( $menuCategoryId ) {
			// We need to re-add the parts we removed above so we can build this properly:
			if ( $groupId ) {
				array_unshift( $segments, $groupId );
			}

			$this->categoryPath( $menuCategoryId, $categoryId, $segments, $vars );
		} elseif ( $menuGroupId ) {
			// We need to re-add the parts we removed above so we can build this properly:
			if ( $groupId ) {
				array_unshift( $segments, $groupId );
			}

			if ( $categoryId && ( $categoryId !== 'groups' ) ) {
				array_unshift( $segments, $categoryId );
			}

			// Group id already exists so lets assume everything that follows is pathing within that group:
			$this->groupPath( $menuCategoryId, $menuGroupId, $segments, $vars );
		} elseif ( $categoryId ) {
			// Not sure where we're going so lets just fallback:
			$vars['action']						=	$categoryId;

			if ( $groupId ) {
				$vars['func']					=	$groupId;

				if ( $count > 0 ) {
					$vars['id']					=	(int) $segments[0];
				}
			}
		}
	}

	/**
	 * @param string|int $categoryId
	 * @param string|int $groupId
	 * @param array      $segments
	 * @param array      $vars
	 */
	private function categoryPath( $categoryId, $groupId, $segments, &$vars )
	{
		$count						=	count( $segments );

		if ( in_array( $groupId, array( 'groups', 'notifications', 'users', 'invites' ), true ) ) {
			$vars['action']			=	$groupId;

			if ( $count > 0 ) {
				$vars['func']		=	$segments[0];

				if ( $count > 1 ) {
					$vars['id']		=	(int) $segments[1];
				}
			}

			if ( $groupId === 'groups' ) {
				$vars['category']	=	(int) $categoryId;
			}
		} else {
			$vars['action']			=	'categories';

			if ( $groupId ) {
				$vars['func']		=	$groupId;
			}

			$vars['id']				=	(int) $categoryId;
		}
	}

	/**
	 * @param string|int $categoryId
	 * @param string|int $groupId
	 * @param array      $segments
	 * @param array      $vars
	 */
	private function groupPath( $categoryId, $groupId, $segments, &$vars )
	{
		$count						=	count( $segments );

		$vars['action']				=	'groups';

		if ( $count > 0 ) {
			$vars['func']			=	$segments[0];

			if ( $vars['func'] === 'invites' ) {
				$vars['group']		=	(int) $groupId;
			} elseif ( $vars['func'] !== 'users' ) {
				$vars['id']			=	(int) $groupId;
			}

			if ( ( $count > 1 ) && in_array( $vars['func'], array( 'notifications', 'users', 'invites' ), true ) ) {
				$vars['action']		=	$vars['func'];

				if ( ( $count > 2 ) && in_array( $vars['func'], array( 'users', 'invites' ), true ) ) {
					$vars['func']	=	$segments[2];
					$vars['id']		=	(int) $segments[1];
				} else {
					$vars['func']	=	$segments[1];
				}
			}
		} else {
			$vars['id']				=	(int) $groupId;
		}
	}
}