<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\GroupJive\Table;

use CB\Plugin\GroupJive\CBGroupJive;
use CBLib\Application\Application;
use CBLib\Database\Table\OrderedTable;
use CBLib\Image\Color;
use CBLib\Language\CBTxt;
use CBLib\Registry\Registry;

defined('CBLIB') or die();

class GroupTable extends OrderedTable
{
	/** @var int  */
	public $id				=	null;
	/** @var int  */
	public $user_id			=	null;
	/** @var int  */
	public $category		=	null;
	/** @var string  */
	public $canvas			=	null;
	/** @var string  */
	public $logo			=	null;
	/** @var string  */
	public $name			=	null;
	/** @var string  */
	public $description		=	null;
	/** @var int  */
	public $type			=	null;
	/** @var string  */
	public $css				=	null;
	/** @var string  */
	public $date			=	null;
	/** @var int  */
	public $published		=	null;
	/** @var int  */
	public $ordering		=	null;
	/** @var string  */
	public $params			=	null;

	/** @var Registry  */
	protected $_input		=	null;
	/** @var Registry  */
	protected $_files		=	null;
	/** @var Registry  */
	protected $_params		=	null;

	/**
	 * Table name in database
	 *
	 * @var string
	 */
	protected $_tbl			=	'#__groupjive_groups';

	/**
	 * Primary key(s) of table
	 *
	 * @var string
	 */
	protected $_tbl_key		=	'id';

	/**
	 * Ordering keys and for each their ordering groups.
	 * E.g.; array( 'ordering' => array( 'tab' ), 'ordering_registration' => array() )
	 * @var array
	 */
	protected $_orderings	=	array( 'ordering' => array( 'category' ) );

	/**
	 * @return bool
	 */
	public function check()
	{
		$clientId			=	Application::Application()->isClient( 'administrator' );
		$isModerator		=	CBGroupJive::isModerator();

		if ( $this->getString( 'name', '' ) === '' ) {
			$this->setError( CBTxt::T( 'Name not specified!' ) );

			return false;
		}

		if ( ! $this->getInt( 'user_id', 0 ) ) {
			$this->setError( CBTxt::T( 'Owner not specified!' ) );

			return false;
		}

		if ( ( $this->getInt( 'category', 0 ) === 0 ) && ( ( ! CBGroupJive::getGlobalParams()->getInt( 'groups_uncategorized', 1 ) ) && ( ! $clientId ) && ( ! $isModerator ) ) ) {
			$this->setError( CBTxt::T( 'Category not specified!' ) );

			return false;
		}

		if ( ! $this->getInt( 'type', 0 ) ) {
			$this->setError( CBTxt::T( 'Type not specified!' ) );

			return false;
		}

		if ( $this->getInt( 'category', 0 ) ) {
			$category		=	$this->category();

			if ( ! $category->getInt( 'id', 0 ) ) {
				$this->setError( CBTxt::T( 'Category does not exist!' ) );

				return false;
			}

			$types			=	cbToArrayOfInt( explode( '|*|', $category->getString( 'types', '1|*|2|*|3|*|4|*|5' ) ) );

			if ( ( ! $clientId ) && ( ! $isModerator ) && $types && ( ! in_array( $this->getInt( 'type', 0 ), $types, true ) ) ) {
				$this->setError( CBTxt::T( 'Type not permitted!' ) );

				return false;
			}
		}

		return true;
	}

	/**
	 * @param bool $updateNulls
	 * @return bool
	 */
	public function store( $updateNulls = false )
	{
		global $_CB_framework, $_PLUGINS;

		$new				=	( ! $this->getInt( 'id', 0 ) );
		$old				=	new self();

		$this->set( 'date', $this->getString( 'date', Application::Database()->getUtcDateTime() ) );

		if ( ! $new ) {
			$old->load( $this->getInt( 'id', 0 ) );

			$_PLUGINS->trigger( 'gj_onBeforeUpdateGroup', array( &$this, $old ) );
		} else {
			$_PLUGINS->trigger( 'gj_onBeforeCreateGroup', array( &$this ) );
		}

		if ( ! parent::store( $updateNulls ) ) {
			return false;
		}

		if ( ! CBGroupJive::uploadImage( 'canvas', $this ) ) {
			return false;
		}

		if ( ! CBGroupJive::uploadImage( 'logo', $this ) ) {
			return false;
		}

		if ( ! parent::store( $updateNulls ) ) {
			return false;
		}

		// If the new owner doesn't match the previous then demote (frontend) or delete (backend) the previous:
		if ( $old->getInt( 'id', 0 ) && ( $old->getInt( 'user_id', 0 ) !== $this->getInt( 'user_id', 0 ) ) ) {
			$previousUser	=	new UserTable();

			$previousUser->load( array( 'user_id' => $old->getInt( 'user_id', 0 ), 'group' => $this->getInt( 'id', 0 ) ) );

			if ( $previousUser->getInt( 'id', 0 ) ) {
				if ( Application::Application()->isClient( 'administrator' ) ) {
					if ( ! $previousUser->delete() ) {
						$this->setError( $previousUser->getError() );

						return false;
					}
				} else {
					$previousUser->set( 'status', 1 );

					if ( ! $previousUser->store() ) {
						$this->setError( $previousUser->getError() );

						return false;
					}
				}
			}
		}

		$user				=	new UserTable();

		$user->load( array( 'user_id' => $this->getInt( 'user_id', 0 ), 'group' => $this->getInt( 'id', 0 ) ) );

		// If the owner doesn't exist or isn't marked owner then create them or promote them to owner:
		if ( ( ! $user->getInt( 'id', 0 ) ) || ( $user->getInt( 'status', 0 ) !== 4 ) ) {
			$user->set( 'user_id', $this->getInt( 'user_id', 0 ) );
			$user->set( 'group', $this->getInt( 'id', 0 ) );
			$user->set( 'status', 4 );

			if ( $user->getError() || ( ! $user->store() ) ) {
				$this->setError( $user->getError() );

				return false;
			}
		}

		// If the category is changed be sure to move the canvas and logo as needed:
		if ( $old->getInt( 'id', 0 ) && ( $this->getString( 'canvas' ) || $this->getString( 'logo' ) ) && ( $old->getInt( 'category', 0 ) !== $this->getInt( 'category', 0 ) ) ) {
			$basePath		=	$_CB_framework->getCfg( 'absolute_path' ) . '/images/comprofiler/plug_cbgroupjive';
			$oldPath		=	$basePath . '/' . $old->getInt( 'category', 0 ) . '/' . $this->getInt( 'id', 0 );
			$newPath		=	$basePath . '/' . $this->getInt( 'category', 0 ) . '/' . $this->getInt( 'id', 0 );

			if ( is_dir( $oldPath ) ) {
				CBGroupJive::createDirectory( $basePath, $this->getInt( 'category', 0 ), $this->getInt( 'id', 0 ) );
				CBGroupJive::copyDirectory( $oldPath, $newPath, true );
			}
		}

		if ( ! $new ) {
			$_PLUGINS->trigger( 'gj_onAfterUpdateGroup', array( $this, $old ) );
		} else {
			$_PLUGINS->trigger( 'gj_onAfterCreateGroup', array( $this ) );
		}

		return true;
	}

	/**
	 * @param null|self $object
	 * @return self|bool
	 */
	public function copy( $object = null )
	{
		global $_CB_framework, $_PLUGINS;

		if ( $object === null ) {
			$object			=	clone $this;
		}

		$old				=	new self();

		$old->load( $object->getInt( 'id', 0 ) );

		$_PLUGINS->trigger( 'gj_onBeforeCopyGroup', array( &$object, $old ) );

		$copy				=	parent::copy( $object );

		if ( ! $copy ) {
			return false;
		}

		// Copy the canvas and logo:
		if ( $object->getString( 'canvas' ) || $object->getString( 'logo' ) ) {
			$basePath		=	$_CB_framework->getCfg( 'absolute_path' ) . '/images/comprofiler/plug_cbgroupjive';
			$oldPath		=	$basePath . '/' . $old->getInt( 'category', 0 ) . '/' . $old->getInt( 'id', 0 );
			$newPath		=	$basePath . '/' . $object->getInt( 'category', 0 ) . '/' . $object->getInt( 'id', 0 );

			if ( is_dir( $oldPath ) ) {
				CBGroupJive::createDirectory( $basePath, $object->getInt( 'category', 0 ), $object->getInt( 'id', 0 ) );
				CBGroupJive::copyDirectory( $oldPath, $newPath );
			}
		}

		$_PLUGINS->trigger( 'gj_onAfterCopyGroup', array( $object, $old ) );

		return $copy;
	}

	/**
	 * @param null|int $id
	 * @return bool
	 */
	public function delete( $id = null )
	{
		global $_CB_framework, $_PLUGINS;

		$_PLUGINS->trigger( 'gj_onBeforeDeleteGroup', array( &$this ) );

		if ( ! parent::delete( $id ) ) {
			return false;
		}

		// Delete users in this group:
		$query				=	'SELECT *'
							.	"\n FROM " . $this->getDbo()->NameQuote( '#__groupjive_users' )
							.	"\n WHERE " . $this->getDbo()->NameQuote( 'group' ) . " = " . $this->getInt( 'id', 0 );
		$this->getDbo()->setQuery( $query );
		$users				=	$this->getDbo()->loadObjectList( null, '\CB\Plugin\GroupJive\Table\UserTable', array( $this->getDbo() ) );

		/** @var UserTable[] $users */
		foreach ( $users as $user ) {
			$user->delete();
		}

		// Delete invites in this group:
		$query				=	'SELECT *'
							.	"\n FROM " . $this->getDbo()->NameQuote( '#__groupjive_invites' )
							.	"\n WHERE " . $this->getDbo()->NameQuote( 'group' ) . " = " . $this->getInt( 'id', 0 );
		$this->getDbo()->setQuery( $query );
		$invites			=	$this->getDbo()->loadObjectList( null, '\CB\Plugin\GroupJive\Table\InviteTable', array( $this->getDbo() ) );

		/** @var InviteTable[] $invites */
		foreach ( $invites as $invite ) {
			$invite->delete();
		}

		// Delete canvas and logo:
		if ( $this->getString( 'canvas' ) || $this->getString( 'logo' ) ) {
			CBGroupJive::deleteDirectory( $_CB_framework->getCfg( 'absolute_path' ) . '/images/comprofiler/plug_cbgroupjive/' . $this->getInt( 'category', 0 ) . '/' . $this->getInt( 'id', 0 ) );
		}

		$_PLUGINS->trigger( 'gj_onAfterDeleteGroup', array( $this ) );

		return true;
	}

	/**
	 * @return Registry
	 */
	public function params()
	{
		if ( ! ( $this->getRaw( '_params' ) instanceof Registry ) ) {
			$this->set( '_params', new Registry( $this->getRaw( 'params' ) ) );
		}

		return $this->getRaw( '_params' );
	}

	/**
	 * @return CategoryTable
	 */
	public function category()
	{
		return CBGroupJive::getCategory( $this->getInt( 'category', 0 ) );
	}

	/**
	 * @return string
	 */
	public function type()
	{
		static $cache				=	array();

		$id							=	$this->getInt( 'type', 0 );

		if ( ! isset( $cache[$id] ) ) {
			switch ( $id ) {
				case 1:
					$cache[$id]		=	CBTxt::T( 'GROUP_TYPE_OPEN', 'Open' );
					break;
				case 2:
					$cache[$id]		=	CBTxt::T( 'GROUP_TYPE_APPROVAL', 'Approval' );
					break;
				case 3:
					$cache[$id]		=	CBTxt::T( 'GROUP_TYPE_PRIVATE', 'Private' );
					break;
				case 4:
					$cache[$id]		=	CBTxt::T( 'GROUP_TYPE_PAGE', 'Page' );
					break;
				case 5:
					$cache[$id]		=	CBTxt::T( 'GROUP_TYPE_INVITE', 'Invite' );
					break;
				default:
					$cache[$id]		=	CBTxt::T( 'Unknown' );
					break;
			}
		}

		return $cache[$id];
	}

	/**
	 * @param bool        $thumbnail
	 * @param bool        $html
	 * @param bool        $linked
	 * @param null|string $classes
	 * @return string
	 */
	public function canvas( $thumbnail = false, $html = true, $linked = false, $classes = null )
	{
		global $_CB_framework;

		static $cache		=	array();

		$id					=	$this->getInt( 'id', 0 ) . $thumbnail;

		if ( ! isset( $cache[$id] ) ) {
			$default		=	CBGroupJive::getGlobalParams()->getString( 'groups_canvas', 'canvas.png' );

			if ( ! $default ) {
				$default	=	'canvas.png';
			}

			$image			=	null;

			if ( $this->getString( 'canvas' ) ) {
				$path		=	'/images/comprofiler/plug_cbgroupjive/' . $this->getInt( 'category', 0 ) . '/' . $this->getInt( 'id', 0 ) . '/' . ( $thumbnail ? 'tn' : null ) . preg_replace( '/[^-a-zA-Z0-9_.]/', '', $this->getString( 'canvas' ) );

				if ( file_exists( $_CB_framework->getCfg( 'absolute_path' ) . $path ) ) {
					$image	=	$_CB_framework->getCfg( 'live_site' ) . $path;
				}
			}

			if ( ! $image ) {
				$image		=	CBGroupJive::getDefaultImage( $default, $thumbnail );
			}

			$cache[$id]		=	$image;
		}

		$canvas				=	$cache[$id];

		if ( ( ! $canvas ) || ( $canvas === 'none' ) ) {
			return null;
		}

		if ( $html ) {
			if ( $canvas === 'color' ) {
				$canvas		=	'<div style="background: linear-gradient( 0deg, ' . htmlspecialchars( Color::stringToHex( $this->getString( 'name' ) ) ) . ' 0%, ' . htmlspecialchars( Color::stringToHex( $this->getString( 'name' ), 0.9 ) ) . ' 100% );" class="cbImgCanvas cbImgCanvasInitial' . ( $thumbnail ? ' cbThumbCanvas' : ' cbFullCanvas' ) . ' gjCanvas gjCanvasDefault' . ( $classes ? ' ' . htmlspecialchars( $classes ) : null ) . '"></div>';
			} else {
				$canvas		=	'<div style="background-image: url(' . htmlspecialchars( $canvas ) . ')" class="cbImgCanvas' . ( $thumbnail ? ' cbThumbCanvas' : ' cbFullCanvas' ) . ' gjCanvas' . ( $this->getString( 'canvas' ) ? ' gjCanvasCustom' : ' gjCanvasDefault' ) . ( $classes ? ' ' . htmlspecialchars( $classes ) : null ) . '"></div>';
			}
		}

		if ( $linked ) {
			$canvas			=	'<a href="' . $_CB_framework->pluginClassUrl( 'cbgroupjive', true, array( 'action' => 'groups', 'func' => 'show', 'id' => $this->getInt( 'id', 0 ) ) ) . '">' . $canvas . '</a>';
		}

		return $canvas;
	}

	/**
	 * @param bool        $thumbnail
	 * @param bool        $html
	 * @param bool        $linked
	 * @param null|string $classes
	 * @return string
	 */
	public function logo( $thumbnail = false, $html = true, $linked = false, $classes = null )
	{
		global $_CB_framework;

		static $cache		=	array();

		$id					=	$this->getInt( 'id', 0 ) . $thumbnail;

		if ( ! isset( $cache[$id] ) ) {
			$default		=	CBGroupJive::getGlobalParams()->getString( 'groups_logo', 'logo.png' );

			if ( ! $default ) {
				$default	=	'logo.png';
			}

			$image			=	null;

			if ( $this->getString( 'logo' ) ) {
				$path		=	'/images/comprofiler/plug_cbgroupjive/' . $this->getInt( 'category', 0 ) . '/' . $this->getInt( 'id', 0 ) . '/' . ( $thumbnail ? 'tn' : null ) . preg_replace( '/[^-a-zA-Z0-9_.]/', '', $this->getString( 'logo' ) );

				if ( file_exists( $_CB_framework->getCfg( 'absolute_path' ) . $path ) ) {
					$image	=	$_CB_framework->getCfg( 'live_site' ) . $path;
				}
			}

			if ( ! $image ) {
				$image		=	CBGroupJive::getDefaultImage( $default, $thumbnail );
			}

			$cache[$id]		=	$image;
		}

		$logo				=	$cache[$id];

		if ( ( ! $logo ) || ( $logo === 'none' ) ) {
			return null;
		}

		if ( $html ) {
			switch ( CBGroupJive::getGlobalParams()->getString( 'groups_logo_style', 'roundedbordered' ) ) {
				case 'rounded':
					$classes	.=	' rounded';
					break;
				case 'roundedbordered':
					$classes	.=	' img-thumbnail';
					break;
				case 'circle':
					$classes	.=	' rounded-circle';
					break;
				case 'circlebordered':
					$classes	.=	' img-thumbnail rounded-circle';
					break;
			}

			$logo			=	'<img alt="' . htmlspecialchars( CBTxt::T( 'Logo' ) ) . '" src="' . htmlspecialchars( $logo ) . '" class="cbImgPict' . ( $thumbnail ? ' cbThumbPict' : ' cbFullPict' ) . ' gjLogo' . ( $this->getString( 'logo' ) ? ' gjLogoCustom' : ' gjLogoDefault' ) . ( $classes ? ' ' . htmlspecialchars( $classes ) : null ) . '" />';
		}

		if ( $linked ) {
			$logo			=	'<a href="' . $_CB_framework->pluginClassUrl( 'cbgroupjive', true, array( 'action' => 'groups', 'func' => 'show', 'id' => $this->getInt( 'id', 0 ) ) ) . '">' . $logo . '</a>';
		}

		return $logo;
	}
}