<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2021 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\Code;

use CBLib\Registry\Registry;
use CB\Database\Table\UserTable;
use CB\Database\Table\FieldTable;
use CBLib\Registry\GetterInterface;

defined('CBLIB') or die();

class CBCodeField
{

	/**
	 * @return Registry
	 */
	static public function getGlobalParams()
	{
		global $_PLUGINS;

		static $params	=	null;

		if ( ! $params ) {
			$plugin		=	$_PLUGINS->getLoadedPlugin( 'user', 'cbcodefield' );
			$params		=	new Registry();

			if ( $plugin ) {
				$params->load( $plugin->params );
			}
		}

		return $params;
	}

	/**
	 * @param FieldTable  $field
	 * @param null|string $file
	 * @param bool|array  $headers
	 * @return null|string
	 */
	static public function getTemplate( $field, $file = null, $headers = array( 'template', 'override' ) )
	{
		global $_CB_framework, $_PLUGINS;

		$plugin							=	$_PLUGINS->getLoadedPlugin( 'user', 'cbcodefield' );

		if ( ! $plugin ) {
			return null;
		}

		static $defaultTemplate			=	null;

		if ( $defaultTemplate === null ) {
			$defaultTemplate			=	self::getGlobalParams()->get( 'general_template', 'default', GetterInterface::STRING );
		}

		$template						=	$field->params->get( 'code_template', null, GetterInterface::STRING );

		if ( ( $template === '' ) || ( $template === null ) || ( $template === '-1' ) ) {
			$template					=	$defaultTemplate;
		}

		if ( ! $template ) {
			$template					=	'default';
		}

		$livePath						=	$_PLUGINS->getPluginLivePath( $plugin );
		$absPath						=	$_PLUGINS->getPluginPath( $plugin );

		$file							=	preg_replace( '/[^-a-zA-Z0-9_]/', '', $file );
		$return							=	null;

		if ( $file ) {
			if ( $headers !== false ) {
				$headers[]				=	$file;
			}

			$php						=	$absPath . '/templates/' . $template . '/' . $file . '.php';

			if ( ! file_exists( $php ) ) {
				$php					=	$absPath . '/templates/default/' . $file . '.php';
			}

			if ( file_exists( $php ) ) {
				$return					=	$php;
			}
		}

		if ( $headers !== false ) {
			static $loaded				=	array();

			$loaded[$template]			=	array();

			// Global CSS File:
			if ( in_array( 'template', $headers ) && ( ! in_array( 'template', $loaded[$template] ) ) ) {
				$global					=	'/templates/' . $template . '/template.css';

				if ( ! file_exists( $absPath . $global ) ) {
					$global				=	'/templates/default/template.css';
				}

				if ( file_exists( $absPath . $global ) ) {
					$_CB_framework->document->addHeadStyleSheet( $livePath . $global );
				}

				$loaded[$template][]	=	'template';
			}

			// File or Custom CSS/JS Headers:
			foreach ( $headers as $header ) {
				if ( in_array( $header, $loaded[$template] ) || in_array( $header, array( 'template', 'override' ) ) ) {
					continue;
				}

				$header					=	preg_replace( '/[^-a-zA-Z0-9_]/', '', $header );

				if ( ! $header ) {
					continue;
				}

				$css					=	'/templates/' . $template . '/' . $header . '.css';
				$js						=	'/templates/' . $template . '/' . $header . '.js';

				if ( ! file_exists( $absPath . $css ) ) {
					$css				=	'/templates/default/' . $header . '.css';
				}

				if ( file_exists( $absPath . $css ) ) {
					$_CB_framework->document->addHeadStyleSheet( $livePath . $css );
				}

				if ( ! file_exists( $absPath . $js ) ) {
					$js					=	'/templates/default/' . $header . '.js';
				}

				if ( file_exists( $absPath . $js ) ) {
					$_CB_framework->document->addHeadScriptUrl( $livePath . $js );
				}

				$loaded[$template][]	=	$header;
			}

			// Override CSS File:
			if ( in_array( 'override', $headers ) && ( ! in_array( 'override', $loaded[$template] ) ) ) {
				$override				=	'/templates/' . $template . '/override.css';

				if ( file_exists( $absPath . $override ) ) {
					$_CB_framework->document->addHeadStyleSheet( $livePath . $override );
				}

				$loaded[$template][]	=	'override';
			}
		}

		return $return;
	}

	/**
	 * Checks if create_function is available
	 *
	 * @return bool
	 */
	static public function canCreateFunction()
	{
		static $cache					=	null;

		if ( $cache === null ) {
			$cache						=	function_exists( 'create_function' );

			if ( $cache && function_exists( 'ini_get' ) ) {
				$disabledFunctions		=	ini_get( 'disabled_functions' );

				if ( $disabledFunctions !== false ) {
					$cache				=	( ! in_array( 'create_function', explode( ',', $disabledFunctions ) ) );
				}
			}

			if ( $cache && version_compare( phpversion(), '7.2.0', '>=' ) ) {
				$cache					=	false;
			}
		}

		return $cache;
	}

	/**
	 * Checks if eval is available
	 *
	 * @return bool
	 */
	static public function canEval()
	{
		static $cache					=	null;

		if ( $cache === null ) {
			$cache						=	true;

			if ( function_exists( 'ini_get' ) ) {
				$disabledFunctions		=	ini_get( 'disabled_functions' );

				if ( $disabledFunctions !== false ) {
					$cache				=	( ! in_array( 'eval', explode( ',', $disabledFunctions ) ) );
				}

				if ( $cache ) {
					$evalDisabled		=	ini_get( 'suhosin.executor.disable_eval' );

					if ( ( $evalDisabled === true ) || ( $evalDisabled === 1 ) ) {
						$cache			=	false;
					}
				}
			}
		}

		return $cache;
	}

	/**
	 * Attempts to execute code from a string
	 *
	 * @param string          $code
	 * @param FieldTable|null $field
	 * @param UserTable|null  $user
	 * @param string|null     $reason
	 * @param mixed           $value
	 * @return string|null
	 */
	static public function outputCode( $code, &$field = null, &$user = null, &$reason = null, &$value = null )
	{
		if ( ! $code ) {
			return null;
		}

		ob_start();
		if ( self::canCreateFunction() ) {
			// PHP 7 has a catchable error so lets catch it if we can and throw it as exception to be caught later:
			if ( version_compare( phpversion(), '7.0.0', '>=' ) ) {
				try {
					$function			=	create_function( '$field,$user,$reason,$value', $code );
					$return				=	$function( $field, $user, $reason, $value );
				} catch ( \Error $e ) {
					throw new \Exception( $e->getMessage() );
				}
			} else {
				$function				=	create_function( '$field,$user,$reason,$value', $code );
				$return					=	$function( $field, $user, $reason, $value );
			}
		} elseif ( self::canEval() ) {
			$function					=	'$function = function( $field, $user, $reason, $value ) { ' . $code . ' };';

			// PHP 7 has a catchable error so lets catch it if we can and throw it as exception to be caught later:
			if ( version_compare( phpversion(), '7.0.0', '>=' ) ) {
				try {
					eval( $function );

					$return				=	$function( $field, $user, $reason, $value );
				} catch ( \Error $e ) {
					throw new \Exception( $e->getMessage() );
				}
			} else {
				eval( $function );

				$return					=	$function( $field, $user, $reason, $value );
			}
		} else {
			// Last resort fallback; this is slow so hopefully it doesn't come to this:
			$temp						=	tmpfile();

			fwrite( $temp, "<?php \n" . $code );

			$tempData					=	stream_get_meta_data( $temp );

			// PHP 7 has a catchable error so lets catch it if we can and throw it as exception to be caught later:
			if ( version_compare( phpversion(), '7.0.0', '>=' ) ) {
				try {
					$return				=	include $tempData['uri'];
				} catch ( \Error $e ) {
					fclose( $temp );

					throw new \Exception( $e->getMessage() );
				}
			} else {
				$return					=	include $tempData['uri'];
			}

			fclose( $temp );

			if ( ! is_string( $return ) ) {
				$return					=	null;
			}
		}
		$echos							=	ob_get_contents();
		ob_end_clean();

		if ( $echos && ( is_string( $return ) || ( $return === '' ) || ( $return === null ) ) ) {
			$return						=	$echos . $return;
		}

		return $return;
	}
}